<?php
// server.php

// Enable error reporting for development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Allow cross-origin requests (CORS) for development.
// In production, you should restrict this to your frontend's domain.
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight OPTIONS requests for CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Database configuration (replace with your actual database credentials)
define('DB_HOST', 'localhost'); // e.g., localhost or your_mysql_host
define('DB_USER', 'root');   // e.g., root or pos_user
define('DB_PASSWORD', ''); // e.g., "" for root or your_password
define('DB_NAME', 'pos_db');

// Establish database connection
$conn = new mysqli(DB_HOST, DB_USER, DB_PASSWORD, DB_NAME);

// Check connection
if ($conn->connect_error) {
    http_response_code(500);
    echo json_encode(["error" => "Connection failed: " . $conn->connect_error]);
    exit();
}

// Ensure tables exist (run once or manually)
function ensureTables($conn) {
    // Products table
    $sql_products = "CREATE TABLE IF NOT EXISTS products (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        price DECIMAL(10, 2) NOT NULL,
        cost DECIMAL(10, 2) NOT NULL,
        stock INT NOT NULL,
        category VARCHAR(255) NOT NULL,
        color VARCHAR(50) NOT NULL
    )";
    if ($conn->query($sql_products) === TRUE) {
        // echo "Table 'products' ensured.\n"; // For console output
    } else {
        error_log("Error creating products table: " . $conn->error);
    }

    // Customers table
    $sql_customers = "CREATE TABLE IF NOT EXISTS customers (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        discounts JSON DEFAULT '{}'
    )";
    if ($conn->query($sql_customers) === TRUE) {
        // echo "Table 'customers' ensured.\n"; // For console output
    } else {
        error_log("Error creating customers table: " . $conn->error);
    }

    // Transactions table
    $sql_transactions = "CREATE TABLE IF NOT EXISTS transactions (
        id INT AUTO_INCREMENT PRIMARY KEY,
        customerId INT,
        customerName VARCHAR(255),
        items JSON NOT NULL,
        total DECIMAL(10, 2) NOT NULL,
        totalCost DECIMAL(10, 2) NOT NULL,
        date DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (customerId) REFERENCES customers(id) ON DELETE SET NULL
    )";
    if ($conn->query($sql_transactions) === TRUE) {
        // echo "Table 'transactions' ensured.\n"; // For console output
    } else {
        error_log("Error creating transactions table: " . $conn->error);
    }
}

ensureTables($conn);

// Get the request method and URI
$method = $_SERVER['REQUEST_METHOD'];
$request_uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$uri_segments = explode('/', trim($request_uri, '/'));

// Assuming API path is /api/resource
// Adjust this if your server setup has a different base path for the API
$api_base_path = 'api'; // Change this if your actual API path is different, e.g., 'pos/api'
$api_base_path_index = array_search($api_base_path, $uri_segments);

if ($api_base_path_index !== false) {
    $resource = $uri_segments[$api_base_path_index + 1] ?? null;
    $id = $uri_segments[$api_base_path_index + 2] ?? null;
    $sub_resource = $uri_segments[$api_base_path_index + 3] ?? null; // For /customers/:id/discounts
} else {
    // Fallback if API base path is not found, assume root level API
    $resource = $uri_segments[0] ?? null;
    $id = $uri_segments[1] ?? null;
    $sub_resource = $uri_segments[2] ?? null;
}


// Read input data for POST/PUT requests
$input = json_decode(file_get_contents('php://input'), true);

switch ($resource) {
    case 'products':
        if ($method === 'GET') {
            if ($id) {
                // Get single product by ID
                $stmt = $conn->prepare("SELECT * FROM products WHERE id = ?");
                $stmt->bind_param("i", $id);
                $stmt->execute();
                $result = $stmt->get_result();
                $product = $result->fetch_assoc();
                if ($product) {
                    echo json_encode($product);
                } else {
                    http_response_code(404);
                    echo json_encode(["error" => "Product not found"]);
                }
                $stmt->close();
            } else {
                // Get all products
                $result = $conn->query("SELECT * FROM products");
                $products = [];
                while ($row = $result->fetch_assoc()) {
                    $products[] = $row;
                }
                echo json_encode($products);
            }
        } elseif ($method === 'POST') {
            $name = $input['name'] ?? '';
            $price = $input['price'] ?? 0.0;
            $cost = $input['cost'] ?? 0.0;
            $stock = $input['stock'] ?? 0;
            $category = $input['category'] ?? '';
            $color = $input['color'] ?? '';

            $stmt = $conn->prepare("INSERT INTO products (name, price, cost, stock, category, color) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("sddiss", $name, $price, $cost, $stock, $category, $color);
            if ($stmt->execute()) {
                http_response_code(201);
                echo json_encode(["id" => $stmt->insert_id, "name" => $name, "price" => $price, "cost" => $cost, "stock" => $stock, "category" => $category, "color" => $color]);
            } else {
                http_response_code(500);
                echo json_encode(["error" => "Failed to add product: " . $stmt->error]);
            }
            $stmt->close();
        } elseif ($method === 'PUT') {
            if ($id) {
                $name = $input['name'] ?? null;
                $price = $input['price'] ?? null;
                $cost = $input['cost'] ?? null;
                $stock = $input['stock'] ?? null;
                $category = $input['category'] ?? null;
                $color = $input['color'] ?? null;

                // Build dynamic UPDATE query based on provided fields
                $set_clauses = [];
                $params = [];
                $types = "";

                if ($name !== null) { $set_clauses[] = "name = ?"; $params[] = $name; $types .= "s"; }
                if ($price !== null) { $set_clauses[] = "price = ?"; $params[] = $price; $types .= "d"; }
                if ($cost !== null) { $set_clauses[] = "cost = ?"; $params[] = $cost; $types .= "d"; }
                if ($stock !== null) { $set_clauses[] = "stock = ?"; $params[] = $stock; $types .= "i"; }
                if ($category !== null) { $set_clauses[] = "category = ?"; $params[] = $category; $types .= "s"; }
                if ($color !== null) { $set_clauses[] = "color = ?"; $params[] = $color; $types .= "s"; }

                if (empty($set_clauses)) {
                    http_response_code(400);
                    echo json_encode(["error" => "No fields provided for update"]);
                    exit();
                }

                $sql = "UPDATE products SET " . implode(', ', $set_clauses) . " WHERE id = ?";
                $params[] = $id;
                $types .= "i";

                $stmt = $conn->prepare($sql);
                if ($stmt === false) {
                    http_response_code(500);
                    echo json_encode(["error" => "Prepare failed: " . $conn->error]);
                    exit();
                }
                $stmt->bind_param($types, ...$params);
                
                if ($stmt->execute()) {
                    if ($stmt->affected_rows > 0) {
                        echo json_encode(["message" => "Product updated successfully"]);
                    } else {
                        http_response_code(404);
                        echo json_encode(["error" => "Product not found or no changes made"]);
                    }
                } else {
                    http_response_code(500);
                    echo json_encode(["error" => "Failed to update product: " . $stmt->error]);
                }
                $stmt->close();
            } else {
                http_response_code(400);
                echo json_encode(["error" => "Product ID is required for PUT request"]);
            }
        } elseif ($method === 'DELETE') {
            if ($id) {
                $stmt = $conn->prepare("DELETE FROM products WHERE id = ?");
                $stmt->bind_param("i", $id);
                if ($stmt->execute()) {
                    if ($stmt->affected_rows > 0) {
                        echo json_encode(["message" => "Product deleted successfully"]);
                    } else {
                        http_response_code(404);
                        echo json_encode(["error" => "Product not found"]);
                    }
                } else {
                    http_response_code(500);
                    echo json_encode(["error" => "Failed to delete product: " . $stmt->error]);
                }
                $stmt->close();
            } else {
                http_response_code(400);
                echo json_encode(["error" => "Product ID is required for DELETE request"]);
            }
        } else {
            http_response_code(405); // Method Not Allowed
            echo json_encode(["error" => "Method not allowed for products"]);
        }
        break;

    case 'customers':
        if ($method === 'GET') {
            $result = $conn->query("SELECT * FROM customers");
            $customers = [];
            while ($row = $result->fetch_assoc()) {
                // Decode JSON string back to array for 'discounts'
                $row['discounts'] = json_decode($row['discounts'], true);
                $customers[] = $row;
            }
            echo json_encode($customers);
        } elseif ($method === 'POST') {
            $name = $input['name'] ?? '';
            $discounts = json_encode($input['discounts'] ?? (object)[]); // Ensure it's a JSON object

            $stmt = $conn->prepare("INSERT INTO customers (name, discounts) VALUES (?, ?)");
            $stmt->bind_param("ss", $name, $discounts);
            if ($stmt->execute()) {
                http_response_code(201);
                echo json_encode(["id" => $stmt->insert_id, "name" => $name, "discounts" => json_decode($discounts)]);
            } else {
                http_response_code(500);
                echo json_encode(["error" => "Failed to add customer: " . $stmt->error]);
            }
            $stmt->close();
        } elseif ($method === 'PUT' && $id && $sub_resource === 'discounts') {
            $discounts = json_encode($input['discounts'] ?? (object)[]); // Expecting full discounts object
            $stmt = $conn->prepare("UPDATE customers SET discounts = ? WHERE id = ?");
            $stmt->bind_param("si", $discounts, $id);
            if ($stmt->execute()) {
                if ($stmt->affected_rows > 0) {
                    echo json_encode(["message" => "Customer discounts updated successfully"]);
                } else {
                    http_response_code(404);
                    echo json_encode(["error" => "Customer not found or no changes made"]);
                }
            } else {
                http_response_code(500);
                echo json_encode(["error" => "Failed to update customer discounts: " . $stmt->error]);
            }
            $stmt->close();
        } else {
            http_response_code(405); // Method Not Allowed
            echo json_encode(["error" => "Method not allowed for customers"]);
        }
        break;

    case 'transactions':
        if ($method === 'GET') {
            $result = $conn->query("SELECT * FROM transactions ORDER BY date DESC");
            $transactions = [];
            while ($row = $result->fetch_assoc()) {
                // Decode JSON string back to array for 'items'
                $row['items'] = json_decode($row['items'], true);
                $transactions[] = $row;
            }
            echo json_encode($transactions);
        } elseif ($method === 'POST') {
            $customerId = $input['customerId'] ?? null;
            $customerName = $input['customerName'] ?? 'Guest';
            $items = json_encode($input['items'] ?? []); // Ensure it's a JSON array
            $total = $input['total'] ?? 0.0;
            $totalCost = $input['totalCost'] ?? 0.0;
            $date = $input['date'] ?? date('Y-m-d H:i:s'); // Use provided date or current timestamp

            $stmt = $conn->prepare("INSERT INTO transactions (customerId, customerName, items, total, totalCost, date) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("isddds", $customerId, $customerName, $items, $total, $totalCost, $date);
            if ($stmt->execute()) {
                http_response_code(201);
                echo json_encode(["id" => $stmt->insert_id, "customerId" => $customerId, "customerName" => $customerName, "items" => json_decode($items), "total" => $total, "totalCost" => $totalCost, "date" => $date]);
            } else {
                http_response_code(500);
                echo json_encode(["error" => "Failed to add transaction: " . $stmt->error]);
            }
            $stmt->close();
        } else {
            http_response_code(405); // Method Not Allowed
            echo json_encode(["error" => "Method not allowed for transactions"]);
        }
        break;

    default:
        http_response_code(404); // Not Found
        echo json_encode(["error" => "Resource not found"]);
        break;
}

$conn->close();
?>
