<?php
session_start();
require_once 'config.php';
require_once 'functions.php';

// التحقق من تسجيل الدخول
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// تحديد الفترة الزمنية
$timePeriod = $_GET['period'] ?? 'this_month';
$startDate = $_GET['start_date'] ?? '';
$endDate = $_GET['end_date'] ?? '';

// حساب نطاق التواريخ
$dateRange = calculateReportDateRange($timePeriod, $startDate, $endDate);

// جلب التقارير
$salesReport = getDailySalesReport($dateRange['start'], $dateRange['end']);
$inventoryReport = getCurrentInventoryReport();
$debtsReport = getCustomerDebtsReport($dateRange['start'], $dateRange['end']);

// حساب إجماليات التقارير
$totalSales = array_sum(array_column($salesReport, 'daily_sales_total'));
$totalProfit = calculateTotalProfit($dateRange['start'], $dateRange['end']);
$totalInventoryValue = getTotalInventoryValue();
$totalDebts = array_sum(array_column($debtsReport, 'credit_balance'));
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>التقارير - نظام نقاط البيع</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body { font-family: 'Tajawal', sans-serif; }
        .report-card { transition: all 0.3s; }
        .report-card:hover { transform: translateY(-5px); box-shadow: 0 10px 20px rgba(0,0,0,0.1); }
        [type="radio"]:checked { background-color: #3b82f6; }
    </style>
</head>
<body class="bg-gray-100">
    <div class="flex h-screen overflow-hidden">
        <?php include 'sidebar.php'; ?>

        <div class="flex-1 overflow-auto">
            <header class="bg-white shadow-sm p-4 flex justify-between items-center">
                <h2 class="text-xl font-bold text-gray-800">التقارير</h2>
                <div class="flex items-center space-x-4">
                    <div class="flex items-center">
                        <span class="mr-2">مرحباً, <?php echo htmlspecialchars($_SESSION['username']); ?></span>
                        <img src="https://ui-avatars.com/api/?name=<?php echo urlencode($_SESSION['username']); ?>&background=random" class="w-8 h-8 rounded-full">
                    </div>
                </div>
            </header>

            <main class="p-4">
                <!-- فلترة التقارير -->
                <div class="bg-white rounded-lg shadow p-4 mb-6">
                    <form method="get" action="reports.php" class="flex flex-col md:flex-row gap-4">
                        <div class="flex-1">
                            <label class="block mb-2 font-medium">الفترة الزمنية:</label>
                            <div class="flex flex-wrap gap-2">
                                <label class="inline-flex items-center">
                                    <input type="radio" name="period" value="today" class="form-radio h-5 w-5 text-blue-600" <?= $timePeriod === 'today' ? 'checked' : '' ?>>
                                    <span class="mr-2">اليوم</span>
                                </label>
                                <label class="inline-flex items-center">
                                    <input type="radio" name="period" value="yesterday" class="form-radio h-5 w-5 text-blue-600" <?= $timePeriod === 'yesterday' ? 'checked' : '' ?>>
                                    <span class="mr-2">أمس</span>
                                </label>
                                <label class="inline-flex items-center">
                                    <input type="radio" name="period" value="this_week" class="form-radio h-5 w-5 text-blue-600" <?= $timePeriod === 'this_week' ? 'checked' : '' ?>>
                                    <span class="mr-2">هذا الأسبوع</span>
                                </label>
                                <label class="inline-flex items-center">
                                    <input type="radio" name="period" value="this_month" class="form-radio h-5 w-5 text-blue-600" <?= $timePeriod === 'this_month' ? 'checked' : '' ?>>
                                    <span class="mr-2">هذا الشهر</span>
                                </label>
                                <label class="inline-flex items-center">
                                    <input type="radio" name="period" value="last_month" class="form-radio h-5 w-5 text-blue-600" <?= $timePeriod === 'last_month' ? 'checked' : '' ?>>
                                    <span class="mr-2">الشهر الماضي</span>
                                </label>
                                <label class="inline-flex items-center">
                                    <input type="radio" name="period" value="custom" class="form-radio h-5 w-5 text-blue-600" <?= $timePeriod === 'custom' ? 'checked' : '' ?>>
                                    <span class="mr-2">مخصص</span>
                                </label>
                            </div>
                        </div>
                        
                        <div class="flex gap-4 <?= $timePeriod === 'custom' ? '' : 'hidden' ?>" id="custom-dates">
                            <div>
                                <label class="block mb-2 font-medium">من تاريخ:</label>
                                <input type="date" name="start_date" value="<?= $startDate ?>" class="p-2 border rounded-lg">
                            </div>
                            <div>
                                <label class="block mb-2 font-medium">إلى تاريخ:</label>
                                <input type="date" name="end_date" value="<?= $endDate ?>" class="p-2 border rounded-lg">
                            </div>
                        </div>
                        
                        <div class="flex items-end">
                            <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                                <i class="fas fa-filter mr-2"></i>تصفية
                            </button>
                        </div>
                    </form>
                </div>

                <!-- بطاقات الإحصائيات -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                    <div class="report-card bg-white rounded-lg shadow p-4">
                        <div class="flex justify-between items-center">
                            <div>
                                <h3 class="text-gray-500">إجمالي المبيعات</h3>
                                <p class="text-2xl font-bold"><?= number_format($totalSales, 2) ?> د.م</p>
                            </div>
                            <div class="p-3 bg-blue-100 rounded-full text-blue-600">
                                <i class="fas fa-shopping-cart text-xl"></i>
                            </div>
                        </div>
                        <div class="mt-2 text-sm text-gray-500">
                            <span>من <?= $dateRange['start'] ?> إلى <?= $dateRange['end'] ?></span>
                        </div>
                    </div>
                    
                    <div class="report-card bg-white rounded-lg shadow p-4">
                        <div class="flex justify-between items-center">
                            <div>
                                <h3 class="text-gray-500">الربح الصافي</h3>
                                <p class="text-2xl font-bold"><?= number_format($totalProfit, 2) ?> د.م</p>
                            </div>
                            <div class="p-3 bg-green-100 rounded-full text-green-600">
                                <i class="fas fa-coins text-xl"></i>
                            </div>
                        </div>
                        <div class="mt-2 text-sm text-gray-500">
                            <span>من <?= $dateRange['start'] ?> إلى <?= $dateRange['end'] ?></span>
                        </div>
                    </div>
                    
                    <div class="report-card bg-white rounded-lg shadow p-4">
                        <div class="flex justify-between items-center">
                            <div>
                                <h3 class="text-gray-500">قيمة المخزون</h3>
                                <p class="text-2xl font-bold"><?= number_format($totalInventoryValue, 2) ?> د.م</p>
                            </div>
                            <div class="p-3 bg-yellow-100 rounded-full text-yellow-600">
                                <i class="fas fa-boxes text-xl"></i>
                            </div>
                        </div>
                        <div class="mt-2 text-sm text-gray-500">
                            <span>حتى <?= date('Y-m-d') ?></span>
                        </div>
                    </div>
                    
                    <div class="report-card bg-white rounded-lg shadow p-4">
                        <div class="flex justify-between items-center">
                            <div>
                                <h3 class="text-gray-500">إجمالي الديون</h3>
                                <p class="text-2xl font-bold"><?= number_format($totalDebts, 2) ?> د.م</p>
                            </div>
                            <div class="p-3 bg-red-100 rounded-full text-red-600">
                                <i class="fas fa-file-invoice-dollar text-xl"></i>
                            </div>
                        </div>
                        <div class="mt-2 text-sm text-gray-500">
                            <span>من <?= $dateRange['start'] ?> إلى <?= $dateRange['end'] ?></span>
                        </div>
                    </div>
                </div>

                <!-- تقرير المبيعات -->
                <div class="bg-white rounded-lg shadow p-4 mb-6">
                    <h3 class="text-lg font-bold mb-4 flex items-center">
                        <i class="fas fa-chart-line text-blue-600 mr-2"></i>
                        تقرير المبيعات
                    </h3>
                    
                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">التاريخ</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">عدد الطلبيات</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">إجمالي المبيعات</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php if (!empty($salesReport)): ?>
                                    <?php foreach ($salesReport as $report): ?>
                                        <tr>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= $report['transaction_date'] ?></td>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= $report['daily_orders_count'] ?></td>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= number_format($report['daily_sales_total'], 2) ?> د.م</td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="3" class="px-6 py-4 text-center text-gray-500">لا توجد بيانات</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                            <tfoot class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-right font-medium text-gray-900">المجموع</th>
                                    <th class="px-6 py-3 text-right font-medium text-gray-900"><?= array_sum(array_column($salesReport, 'daily_orders_count')) ?></th>
                                    <th class="px-6 py-3 text-right font-medium text-gray-900"><?= number_format($totalSales, 2) ?> د.م</th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>

                <!-- تقرير المخزون -->
                <div class="bg-white rounded-lg shadow p-4 mb-6">
                    <h3 class="text-lg font-bold mb-4 flex items-center">
                        <i class="fas fa-boxes text-yellow-600 mr-2"></i>
                        تقرير المخزون
                    </h3>
                    
                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">اسم المنتج</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">الكمية</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">سعر الشراء</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">القيمة</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">المورد</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">التصنيف</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php if (!empty($inventoryReport)): ?>
                                    <?php foreach ($inventoryReport as $product): ?>
                                        <tr>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= htmlspecialchars($product['product_name']) ?></td>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= $product['stock_quantity'] ?></td>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= number_format($product['unit_price'], 2) ?> د.م</td>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= number_format($product['unit_price'] * $product['stock_quantity'], 2) ?> د.م</td>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= htmlspecialchars($product['supplier']) ?></td>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= htmlspecialchars($product['category']) ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="6" class="px-6 py-4 text-center text-gray-500">لا توجد بيانات</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                            <tfoot class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-right font-medium text-gray-900" colspan="3">المجموع</th>
                                    <th class="px-6 py-3 text-right font-medium text-gray-900"><?= number_format($totalInventoryValue, 2) ?> د.م</th>
                                    <th colspan="2"></th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>

                <!-- تقرير الديون -->
                <div class="bg-white rounded-lg shadow p-4">
                    <h3 class="text-lg font-bold mb-4 flex items-center">
                        <i class="fas fa-file-invoice-dollar text-red-600 mr-2"></i>
                        تقرير الديون
                    </h3>
                    
                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">اسم الزبون</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">رصيد الدين</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">عدد الطلبيات غير المسددة</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php if (!empty($debtsReport)): ?>
                                    <?php foreach ($debtsReport as $debt): ?>
                                        <tr>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= htmlspecialchars($debt['name']) ?></td>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= number_format($debt['credit_balance'], 2) ?> د.م</td>
                                            <td class="px-6 py-4 whitespace-nowrap"><?= $debt['unpaid_orders'] ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="3" class="px-6 py-4 text-center text-gray-500">لا توجد ديون</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                            <tfoot class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-right font-medium text-gray-900">المجموع</th>
                                    <th class="px-6 py-3 text-right font-medium text-gray-900"><?= number_format($totalDebts, 2) ?> د.م</th>
                                    <th class="px-6 py-3 text-right font-medium text-gray-900"><?= array_sum(array_column($debtsReport, 'unpaid_orders')) ?></th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
        $(document).ready(function() {
            // إظهار/إخفاء حقول التواريخ المخصصة
            $('input[name="period"]').change(function() {
                if ($(this).val() === 'custom') {
                    $('#custom-dates').removeClass('hidden');
                } else {
                    $('#custom-dates').addClass('hidden');
                }
            });
        });
    </script>
</body>
</html>