<?php
session_start();
require_once 'config.php';
require_once 'functions.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$products = getProducts();
$categories = getCategories();
$customers = getCustomers();
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>نظام نقاط البيع</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body { font-family: 'Tajawal', sans-serif; }
        .product-card:hover { transform: scale(1.03); box-shadow: 0 10px 20px rgba(0,0,0,0.1); }
        .cart-item { transition: all 0.2s; }
        .cart-item:hover { background-color: #f8fafc; }
        [type="radio"]:checked { background-color: #3b82f6; }
    </style>
</head>
<body class="bg-gray-100">
    <div class="flex h-screen overflow-hidden">
        <?php include 'sidebar.php'; ?>

        <div class="flex-1 overflow-auto">
            <header class="bg-white shadow-sm p-4 flex justify-between items-center">
                <h2 class="text-xl font-bold text-gray-800">نقاط البيع</h2>
                <div class="flex items-center space-x-4">
                    <a href="debts.php" class="bg-red-500 text-white px-4 py-2 rounded-lg hover:bg-red-600">
                        <i class="fas fa-file-invoice-dollar mr-2"></i>إدارة الديون
                    </a>
                    <div class="flex items-center">
                        <span class="mr-2">مرحباً, <?php echo htmlspecialchars($_SESSION['username']); ?></span>
                        <img src="https://ui-avatars.com/api/?name=<?php echo urlencode($_SESSION['username']); ?>&background=random" class="w-8 h-8 rounded-full">
                    </div>
                </div>
            </header>

            <main class="flex flex-col md:flex-row h-full">
                <div class="flex-1 p-4 overflow-auto">
                    <div class="mb-4">
                        <input type="text" id="search-products" placeholder="ابحث عن منتج..." class="w-full p-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>

                    <div class="flex flex-wrap gap-2 mb-4">
                        <button class="category-btn px-4 py-2 rounded-full bg-blue-600 text-white" data-category="all">الكل</button>
                        <?php foreach ($categories as $category_item): ?>
                        <button class="category-btn px-4 py-2 rounded-full bg-gray-200 hover:bg-blue-100" data-category="<?php echo htmlspecialchars($category_item); ?>">
                            <?php echo htmlspecialchars($category_item); ?>
                        </button>
                        <?php endforeach; ?>
                    </div>

                    <div class="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 gap-4" id="products-grid">
                        <?php foreach ($products as $product): ?>
                        <div class="product-card bg-white rounded-lg shadow overflow-hidden cursor-pointer transition duration-300" 
                             data-id="<?php echo $product['id']; ?>"
                             data-name="<?php echo htmlspecialchars($product['name']); ?>"
                             data-price="<?php echo $product['price']; ?>"
                             data-stock="<?php echo $product['stock']; ?>"
                             data-category="<?php echo htmlspecialchars($product['category']); ?>">
                            <div class="h-32 flex items-center justify-center <?php echo $product['stock'] > 0 ? 'bg-blue-500' : 'bg-gray-300'; ?>">
                                <span class="text-white text-xl font-bold"><?php echo $product['stock']; ?></span>
                            </div>
                            <div class="p-3">
                                <h3 class="font-bold text-center"><?php echo htmlspecialchars($product['name']); ?></h3>
                                <p class="text-center text-gray-700"><?php echo number_format($product['price'], 2); ?> د.م</p>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <div class="w-full md:w-96 bg-white border-l p-4 flex flex-col">
                    <div class="flex justify-between items-center mb-4">
                        <h3 class="text-xl font-bold">سلة التسوق</h3>
                        <button id="clear-cart" class="text-red-600 hover:text-red-800">
                            <i class="fas fa-trash-alt"></i> إفراغ السلة
                        </button>
                    </div>

                    <div class="mb-4 p-3 bg-gray-50 rounded-lg">
                        <div class="flex justify-between items-center mb-2">
                            <span class="font-medium">الزبون:</span>
                            <span id="customer-name">زبون عابر</span>
                        </div>
                        <div class="flex space-x-2">
                            <button id="select-customer" class="flex-1 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                                <i class="fas fa-user-plus mr-2"></i> تحديد زبون
                            </button>
                            <button id="view-customer-credit" class="px-3 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700" title="عرض رصيد الزبون">
                                <i class="fas fa-coins"></i>
                            </button>
                        </div>
                    </div>

                    <div class="flex-1 overflow-auto mb-4" id="cart-items">
                        <p class="text-center text-gray-500 py-10">السلة فارغة</p>
                    </div>

                    <div class="border-t pt-4">
                        <div class="flex justify-between mb-2">
                            <span>المجموع الفرعي:</span>
                            <span id="subtotal">0.00 د.م</span>
                        </div>
                        <div class="flex justify-between mb-2">
                            <span>الخصم:</span>
                            <span id="discount-total">0.00 د.م</span>
                        </div>
                        <div class="flex justify-between font-bold text-lg mb-4">
                            <span>الإجمالي:</span>
                            <span id="total">0.00 د.م</span>
                        </div>
                        <button id="checkout" class="w-full py-3 bg-green-600 text-white rounded-lg hover:bg-green-700">
                            <i class="fas fa-cash-register mr-2"></i> إتمام البيع
                        </button>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <div id="customer-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center hidden z-50">
        <div class="bg-white rounded-lg p-6 w-full max-w-md">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-xl font-bold">اختيار الزبون</h3>
                <button id="close-customer-modal" class="text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="mb-4">
                <input type="text" id="search-customers" placeholder="ابحث عن زبون..." class="w-full p-2 border rounded-lg">
            </div>
            <div class="max-h-60 overflow-auto mb-4" id="customers-list">
                <?php foreach ($customers as $customer): ?>
                <div class="customer-item p-3 border-b cursor-pointer hover:bg-gray-50" 
                     data-id="<?php echo $customer['id']; ?>"
                     data-name="<?php echo htmlspecialchars($customer['name']); ?>"
                     data-discounts='<?php echo json_encode($customer['discounts'] ?? (object)[], JSON_UNESCAPED_UNICODE); ?>'
                     data-credit="<?php echo $customer['credit_balance'] ?? 0; ?>">
                    <div class="flex justify-between items-center">
                        <span><?php echo htmlspecialchars($customer['name']); ?></span>
                        <span class="text-sm <?php echo ($customer['credit_balance'] ?? 0) > 0 ? 'text-red-600' : 'text-green-600'; ?>">
                            <?php echo number_format($customer['credit_balance'] ?? 0, 2); ?> د.م
                        </span>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <div class="mb-4">
                <input type="text" id="new-customer-name" placeholder="اسم الزبون الجديد" class="w-full p-2 border rounded-lg">
            </div>
            <button id="add-customer" class="w-full py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                <i class="fas fa-plus mr-2"></i> إضافة زبون جديد
            </button>
        </div>
    </div>

    <div id="payment-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center hidden z-50">
        <div class="bg-white rounded-lg p-6 w-full max-w-md">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-xl font-bold">طريقة الدفع</h3>
                <button id="close-payment-modal" class="text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <div class="mb-4">
                <label class="block mb-2 font-medium">اختر طريقة الدفع:</label>
                <div class="flex flex-col space-y-2">
                    <label class="inline-flex items-center">
                        <input type="radio" name="payment_method" value="cash" class="form-radio h-5 w-5 text-blue-600" checked>
                        <span class="mr-2">كاش</span>
                    </label>
                    <label class="inline-flex items-center">
                        <input type="radio" name="payment_method" value="credit" class="form-radio h-5 w-5 text-blue-600">
                        <span class="mr-2">دين</span>
                    </label>
                </div>
            </div>
            
            <div id="credit-section" class="hidden">
                <div class="mb-4">
                    <label class="block mb-2 font-medium">رصيد الزبون الحالي:</label>
                    <div id="current-credit" class="p-2 bg-gray-100 rounded-lg font-bold text-red-600">0.00 د.م</div>
                </div>
                
                <div class="mb-4">
                    <label class="block mb-2 font-medium">الرصيد بعد الشراء:</label>
                    <div id="new-credit" class="p-2 bg-gray-100 rounded-lg font-bold text-red-600">0.00 د.م</div>
                </div>
                
                <div class="mb-4">
                    <label class="block mb-2 font-medium">ملاحظات الدين:</label>
                    <textarea id="credit_notes" class="w-full p-2 border rounded-lg" placeholder="أي ملاحظات حول الدين..."></textarea>
                </div>
            </div>
            
            <div class="flex space-x-2">
                <button id="confirm-payment" class="flex-1 py-3 bg-green-600 text-white rounded-lg hover:bg-green-700">
                    تأكيد العملية
                </button>
                <button id="cancel-payment" class="px-4 py-3 bg-gray-500 text-white rounded-lg hover:bg-gray-600">
                    إلغاء
                </button>
            </div>
        </div>
    </div>

    <div id="credit-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center hidden z-50">
        <div class="bg-white rounded-lg p-6 w-full max-w-md">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-xl font-bold">رصيد الزبون</h3>
                <button id="close-credit-modal" class="text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="mb-4">
                <h4 id="credit-customer-name" class="font-bold text-lg"></h4>
                <div class="p-3 bg-gray-100 rounded-lg mt-2">
                    <div class="flex justify-between items-center">
                        <span class="font-medium">الرصيد الحالي:</span>
                        <span id="credit-balance" class="font-bold text-xl">0.00 د.م</span>
                    </div>
                </div>
            </div>
            <div class="border-t pt-4">
                <button id="close-credit-modal-btn" class="w-full py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                    موافق
                </button>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
        $(document).ready(function() {
            let cart = [];
            let selectedCustomer = null;
            let discountTimer;

            // تحديث السلة
            function updateCart() {
                const cartItems = $('#cart-items');
                cartItems.empty();

                let subtotal = 0;
                let totalDiscount = 0;

                if (cart.length === 0) {
                    cartItems.html('<p class="text-center text-gray-500 py-10">السلة فارغة</p>');
                } else {
                    cart.forEach(item => {
                        const itemDiscount = parseFloat(item.discount) || 0; 
                        const itemTotal = (item.price * item.quantity) - (itemDiscount * item.quantity);
                        
                        subtotal += (item.price * item.quantity);
                        totalDiscount += (itemDiscount * item.quantity);
                        
                        const cartItem = $(`
                            <div class="cart-item p-3 border-b">
                                <div class="flex justify-between items-center">
                                    <div>
                                        <h4 class="font-medium">${item.name}</h4>
                                        <p class="text-sm text-gray-600">${item.price.toFixed(2)} د.م × ${item.quantity}</p>
                                    </div>
                                    <span class="font-medium">${itemTotal.toFixed(2)} د.م</span>
                                </div>
                                <div class="mt-2 flex items-center justify-between">
                                    <div class="flex items-center">
                                        <button class="quantity-btn decrease px-2 py-1 bg-gray-200 rounded" data-id="${item.id}">
                                            <i class="fas fa-minus"></i>
                                        </button>
                                        <span class="mx-2">${item.quantity}</span>
                                        <button class="quantity-btn increase px-2 py-1 bg-gray-200 rounded" data-id="${item.id}">
                                            <i class="fas fa-plus"></i>
                                        </button>
                                    </div>
                                    <div class="flex items-center">
                                        <input type="number" class="discount-input w-20 p-1 border rounded text-left" 
                                               data-id="${item.id}" value="${itemDiscount.toFixed(2)}" min="0" step="0.01">
                                        <span class="text-sm text-gray-600 mr-2">د.م</span>
                                    </div>
                                </div>
                            </div>
                        `);
                        
                        cartItems.append(cartItem);
                    });
                }

                const total = subtotal - totalDiscount;
                $('#subtotal').text(subtotal.toFixed(2) + ' د.م');
                $('#discount-total').text(totalDiscount.toFixed(2) + ' د.م');
                $('#total').text(total.toFixed(2) + ' د.م');
                
                if (selectedCustomer) {
                    $('#current-credit').text((selectedCustomer.credit || 0).toFixed(2) + ' د.م');
                    $('#new-credit').text(((selectedCustomer.credit || 0) + total).toFixed(2) + ' د.م');
                }
            }

            // تطبيق خصومات الزبون
            function applyCustomerDiscounts() {
                if (!selectedCustomer?.discounts) return;
                
                cart = cart.map(item => {
                    const discount = selectedCustomer.discounts[item.id];
                    return discount !== undefined ? 
                        {...item, discount: parseFloat(discount) || 0} : 
                        item;
                });
                
                updateCart();
            }

            // حفظ الخصم
            function saveCustomerDiscount(productId, discountAmount) {
                clearTimeout(discountTimer);
                
                if (!selectedCustomer) {
                    alert('الرجاء تحديد زبون أولاً');
                    return;
                }
                
                const discountValue = parseFloat(discountAmount) || 0;
                selectedCustomer.discounts = {
                    ...selectedCustomer.discounts,
                    [productId]: discountValue
                };
                
                cart = cart.map(item => 
                    item.id === productId ? {...item, discount: discountValue} : item
                );
                
                updateCart();
                
                discountTimer = setTimeout(() => {
                    $.ajax({
                        url: 'api/update_discount.php',
                        method: 'POST',
                        contentType: 'application/json',
                        data: JSON.stringify({
                            customer_id: selectedCustomer.id,
                            product_id: productId,
                            discount: discountValue
                        }),
                        success: function(response) {
                            if (!response?.success) {
                                console.error('Failed to save discount:', response?.message);
                                alert('تم تطبيق الخصم محلياً ولكن فشل حفظه على الخادم');
                            }
                        },
                        error: function(xhr, status, error) {
                            console.error('Error saving discount:', status, error);
                        }
                    });
                }, 800);
            }

            // إدارة الزبائن
            function handleSelectCustomer(customer) {
                selectedCustomer = {
                    id: $(customer).data('id'),
                    name: $(customer).data('name'),
                    discounts: $(customer).data('discounts') || {},
                    credit: parseFloat($(customer).data('credit')) || 0
                };
                
                $('#customer-name').text(selectedCustomer.name);
                $('#customer-modal').addClass('hidden');
                applyCustomerDiscounts();
            }

            // أحداث المنتجات
            $('.product-card').click(function() {
                const id = $(this).data('id');
                const name = $(this).data('name');
                const price = parseFloat($(this).data('price'));
                const stock = parseInt($(this).data('stock'));
                
                if (stock <= 0) {
                    alert('هذا المنتج غير متوفر في المخزن!');
                    return;
                }

                const discount = selectedCustomer?.discounts?.[id] || 0;
                const existingItem = cart.find(item => item.id === id);
                
                if (existingItem) {
                    existingItem.quantity += 1;
                } else {
                    cart.push({
                        id: id,
                        name: name,
                        price: price,
                        quantity: 1,
                        discount: discount
                    });
                }

                updateCart();
            });

            // أحداث السلة
            $(document).on('click', '.quantity-btn.decrease', function() {
                const id = $(this).data('id');
                const item = cart.find(item => item.id === id);
                
                if (item.quantity > 1) {
                    item.quantity -= 1;
                } else {
                    cart = cart.filter(item => item.id !== id);
                }
                
                updateCart();
            });

            $(document).on('click', '.quantity-btn.increase', function() {
                const id = $(this).data('id');
                const item = cart.find(item => item.id === id);
                item.quantity += 1;
                updateCart();
            });

            $(document).on('change', '.discount-input', function() {
                const id = $(this).data('id');
                const discountAmount = $(this).val();
                saveCustomerDiscount(id, discountAmount);
            });

            $('#clear-cart').click(function() {
                if (cart.length > 0 && confirm('هل أنت متأكد من أنك تريد إفراغ السلة؟')) {
                    cart = [];
                    updateCart();
                }
            });

            // أحداث الزبائن
            $('#select-customer').click(function() {
                $('#customer-modal').removeClass('hidden');
            });

            $('#close-customer-modal').click(function() {
                $('#customer-modal').addClass('hidden');
            });

            $(document).on('click', '.customer-item', function() {
                handleSelectCustomer(this);
            });

            $('#add-customer').click(function() {
                const name = $('#new-customer-name').val().trim();
                
                if (name === '') {
                    alert('الرجاء إدخال اسم الزبون');
                    return;
                }

                $.ajax({
                    url: 'api/customers.php',
                    method: 'POST',
                    contentType: 'application/json',
                    data: JSON.stringify({ 
                        name: name, 
                        discounts: {},
                        credit_balance: 0
                    }),
                    success: function(response) {
                        if (response?.success) {
                            const newCustomer = $(`
                                <div class="customer-item p-3 border-b cursor-pointer hover:bg-gray-50" 
                                     data-id="${response.data.id}"
                                     data-name="${name}"
                                     data-discounts='{}'
                                     data-credit="0">
                                    <div class="flex justify-between items-center">
                                        <span>${name}</span>
                                        <span class="text-sm text-green-600">0.00 د.م</span>
                                    </div>
                                </div>
                            `);
                            $('#customers-list').prepend(newCustomer);
                            $('#new-customer-name').val('');
                            alert('تم إضافة الزبون بنجاح');
                        } else {
                            alert(response?.message || 'فشل في إضافة الزبون');
                        }
                    },
                    error: function(xhr) {
                        alert('حدث خطأ في الاتصال: ' + xhr.statusText);
                    }
                });
            });

            // عرض رصيد الزبون
            $('#view-customer-credit').click(function() {
                if (!selectedCustomer) {
                    alert('الرجاء تحديد زبون أولاً');
                    return;
                }
                
                $('#credit-customer-name').text(selectedCustomer.name);
                $('#credit-balance').text((selectedCustomer.credit || 0).toFixed(2) + ' د.م');
                $('#credit-modal').removeClass('hidden');
            });

            $('#close-credit-modal, #close-credit-modal-btn').click(function() {
                $('#credit-modal').addClass('hidden');
            });

            // إدارة الدفع
            $('#checkout').click(function() {
                if (cart.length === 0) {
                    alert('الرجاء إضافة منتجات إلى السلة أولاً');
                    return;
                }
                
                const total = parseFloat($('#total').text().replace(' د.م', '')) || 0;
                if (selectedCustomer) {
                    $('#current-credit').text((selectedCustomer.credit || 0).toFixed(2) + ' د.م');
                    $('#new-credit').text(((selectedCustomer.credit || 0) + total).toFixed(2) + ' د.م');
                }
                
                $('#payment-modal').removeClass('hidden');
            });

            $('input[name="payment_method"]').change(function() {
                if ($(this).val() === 'credit') {
                    if (!selectedCustomer) {
                        alert('الرجاء تحديد زبون أولاً');
                        $(this).prop('checked', false);
                        $('input[name="payment_method"][value="cash"]').prop('checked', true);
                        return;
                    }
                    
                    const total = parseFloat($('#total').text().replace(' د.م', '')) || 0;
                    $('#current-credit').text((selectedCustomer.credit || 0).toFixed(2) + ' د.م');
                    $('#new-credit').text(((selectedCustomer.credit || 0) + total).toFixed(2) + ' د.م');
                    $('#credit-section').removeClass('hidden');
                } else {
                    $('#credit-section').addClass('hidden');
                }
            });

            $('#close-payment-modal, #cancel-payment').click(function() {
                $('#payment-modal').addClass('hidden');
            });

            $('#confirm-payment').click(function() {
    const paymentMethod = $('input[name="payment_method"]:checked').val();
    const creditNotes = $('#credit_notes').val();
    const total = parseFloat($('#total').text().replace(' د.م', '')) || 0;
    
    if (paymentMethod === 'credit' && (!selectedCustomer || !selectedCustomer.id)) {
        alert('لا يمكن إتمام عملية الدين بدون تحديد زبون');
        return;
    }

    const transactionData = {
        action: 'create_transaction',
        customer_id: selectedCustomer?.id || null,
        customer_name: selectedCustomer?.name || 'زبون عابر',
        items: cart.map(item => ({
            id: item.id,
            name: item.name,
            quantity: item.quantity,
            price: item.price,
            discount: parseFloat(item.discount) || 0
        })),
        total: total,
        payment_method: paymentMethod,
        status: paymentMethod === 'cash' ? 'paid' : 'unpaid',
        notes: creditNotes
    };

    // إظهار حالة التحميل
    const $confirmBtn = $(this);
    $confirmBtn.html('<i class="fas fa-spinner fa-spin"></i> جاري المعالجة...').prop('disabled', true);

    $.ajax({
        url: 'transactions.php',
        method: 'POST',
        contentType: 'application/json',
        data: JSON.stringify(transactionData),
        dataType: 'json',
        success: function(response) {
            if (response && response.success) {
                let msg = 'تمت العملية بنجاح!';
                
                // عرض رقم الفاتورة إذا كان موجوداً
                if (response.transaction_id) {
                    msg += ` رقم الفاتورة: ${response.transaction_id}`;
                } else if (response.id) { // إذا كان اسم الحقل مختلفاً
                    msg += ` رقم الفاتورة: ${response.id}`;
                }
                
                if (paymentMethod === 'credit' && selectedCustomer) {
                    const newCredit = response.remaining_credit || (selectedCustomer.credit + total);
                    msg += `\nرصيد الزبون الجديد: ${parseFloat(newCredit).toFixed(2)} د.م`;
                    selectedCustomer.credit = parseFloat(newCredit);
                }
                
                alert(msg);
                cart = [];
                updateCart();
                $('#payment-modal').addClass('hidden');
                
                // التحديث التلقائي فقط إذا لزم الأمر
                if (response.update_required) {
                    window.location.reload();
                }
            } else {
                const errorMsg = response?.message || 'فشل في إتمام العملية بدون رسالة خطأ';
                alert(errorMsg);
            }
        },
        error: function(xhr, status, error) {
            let errorMsg = 'حدث خطأ في الاتصال';
            if (xhr.responseText) {
                try {
                    const serverResponse = JSON.parse(xhr.responseText);
                    errorMsg = serverResponse.message || errorMsg;
                } catch (e) {
                    errorMsg += `: ${xhr.responseText.substring(0, 100)}`;
                }
            }
            alert(errorMsg);
            console.error('تفاصيل الخطأ:', status, error, xhr.responseText);
        },
        complete: function() {
            $confirmBtn.html('تأكيد العملية').prop('disabled', false);
        }
    });
});
            // تصفية المنتجات
            $('#search-products').on('input', function() {
                const searchTerm = $(this).val().toLowerCase();
                $('.product-card').each(function() {
                    const productName = $(this).data('name').toLowerCase();
                    $(this).toggle(productName.includes(searchTerm));
                });
            });

            $('.category-btn').click(function() {
                const category = $(this).data('category');
                $('.category-btn').removeClass('bg-blue-600 text-white').addClass('bg-gray-200 hover:bg-blue-100');
                $(this).removeClass('bg-gray-200 hover:bg-blue-100').addClass('bg-blue-600 text-white');
                
                $('.product-card').each(function() {
                    const productCategory = $(this).data('category');
                    $(this).toggle(category === 'all' || productCategory === category);
                });
            });

            $('#search-customers').on('input', function() {
                const searchTerm = $(this).val().toLowerCase();
                $('.customer-item').each(function() {
                    const customerName = $(this).data('name').toLowerCase();
                    $(this).toggle(customerName.includes(searchTerm));
                });
            });
        });
    </script>
</body>
</html>