<?php
require_once 'config.php';

// الاتصال بقاعدة البيانات
function getDBConnection() {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4",
                PDO::ATTR_EMULATE_PREPARES => false
            ];
            
            $pdo = new PDO(
                "mysql:host=".DB_HOST.";dbname=".DB_NAME.";charset=utf8mb4", 
                DB_USER, 
                DB_PASS,
                $options
            );
        } catch (PDOException $e) {
            error_log("فشل الاتصال بقاعدة البيانات: " . $e->getMessage());
            die("فشل الاتصال بقاعدة البيانات. الرجاء المحاولة لاحقاً.");
        }
    }
    
    return $pdo;
}

// جلب تاريخ آخر معاملة للزبون
function getLastOrderDate($customer_id) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT order_date 
            FROM transactions 
            WHERE customer_id = ? 
            ORDER BY order_date DESC 
            LIMIT 1
        ");
        $stmt->execute([$customer_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? $result['order_date'] : null;
    } catch (PDOException $e) {
        error_log("Error fetching last order date: " . $e->getMessage());
        return null;
    }
}

// جلب جميع المنتجات مع تحسين الأداء
function getProducts() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT p.*, 
                   (SELECT COUNT(*) FROM transaction_items WHERE product_id = p.id) AS sales_count
            FROM products p 
            ORDER BY p.name ASC
        ");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching products: " . $e->getMessage());
        return [];
    }
}

// جلب جميع التصنيفات مع تحسين الأداء
function getCategories() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT DISTINCT category 
            FROM products 
            WHERE category IS NOT NULL AND category != ''
            ORDER BY category ASC
        ");
        $categories = $stmt->fetchAll(PDO::FETCH_COLUMN);
        return $categories ?: ['غير مصنف'];
    } catch (PDOException $e) {
        error_log("Error fetching categories: " . $e->getMessage());
        return ['غير مصنف'];
    }
}

// جلب جميع الزبائن مع خصوماتهم - النسخة المحسنة
function getCustomers() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT id, name, discounts, credit_balance,
                   (SELECT COUNT(*) FROM transactions WHERE customer_id = customers.id) AS orders_count
            FROM customers 
            ORDER BY name ASC
        ");
        $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($customers as &$customer) {
            $customer['discounts'] = json_decode($customer['discounts'], true) ?? [];
            $customer['credit_balance'] = (float)$customer['credit_balance'];
        }
        
        return $customers;
    } catch (PDOException $e) {
        error_log("Error fetching customers: " . $e->getMessage());
        return [];
    }
}

// جلب زبون معين بخصوماته - النسخة المحسنة
function getCustomer($customerId) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM customers 
            WHERE id = ?
        ");
        $stmt->execute([$customerId]);
        $customer = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($customer) {
            $customer['discounts'] = json_decode($customer['discounts'], true) ?? [];
            $customer['credit_balance'] = (float)$customer['credit_balance'];
        }
        
        return $customer;
    } catch (PDOException $e) {
        error_log("Error fetching customer: " . $e->getMessage());
        return null;
    }
}

// تحديث خصومات الزبون - النسخة المحسنة
function updateCustomerDiscounts($customerId, $discounts) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            UPDATE customers 
            SET discounts = ? 
            WHERE id = ?
        ");
        $success = $stmt->execute([
            json_encode($discounts, JSON_UNESCAPED_UNICODE),
            $customerId
        ]);
        
        if ($success && $stmt->rowCount() > 0) {
            return true;
        } else {
            error_log("Failed to update customer discounts. Customer ID: $customerId");
            return false;
        }
    } catch (PDOException $e) {
        error_log("Error updating customer discounts: " . $e->getMessage());
        return false;
    }
}

// جلب أحدث المعاملات مع تحسين الأداء
function getRecentTransactions($limit = 5) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT t.*, COUNT(ti.id) AS items_count
            FROM transactions t
            LEFT JOIN transaction_items ti ON t.id = ti.transaction_id
            GROUP BY t.id
            ORDER BY t.date DESC 
            LIMIT ?
        ");
        $stmt->bindValue(1, $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching transactions: " . $e->getMessage());
        return [];
    }
}

// جلب إحصائيات لوحة التحكم - النسخة المحسنة
function getDashboardStats() {
    $pdo = getDBConnection();
    
    $stats = [
        'total_sales' => 0,
        'net_profit' => 0,
        'total_orders' => 0,
        'total_products_sold' => 0,
        'active_customers' => 0
    ];
    
    try {
        // إجمالي المبيعات
        $stmt = $pdo->query("
            SELECT COALESCE(SUM(total), 0) AS total_sales 
            FROM transactions
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stats['total_sales'] = (float)$result['total_sales'];
        
        // إجمالي الربح الصافي
        $stmt = $pdo->query("
            SELECT COALESCE(SUM(total - total_cost), 0) AS net_profit 
            FROM transactions
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stats['net_profit'] = (float)$result['net_profit'];
        
        // إجمالي عدد الطلبات
        $stmt = $pdo->query("
            SELECT COUNT(*) AS total_orders 
            FROM transactions
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stats['total_orders'] = (int)$result['total_orders'];
        
        // إجمالي المنتجات المباعة
        $stmt = $pdo->query("
            SELECT COALESCE(SUM(quantity), 0) AS total_products_sold 
            FROM transaction_items
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stats['total_products_sold'] = (int)$result['total_products_sold'];
        
        // عدد الزبائن النشطين
        $stmt = $pdo->query("
            SELECT COUNT(DISTINCT customer_id) AS active_customers 
            FROM transactions 
            WHERE customer_id IS NOT NULL
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stats['active_customers'] = (int)$result['active_customers'];
        
    } catch (PDOException $e) {
        error_log("Error fetching dashboard stats: " . $e->getMessage());
    }
    
    return $stats;
}

// جلب المنتجات الأكثر مبيعاً - النسخة المحسنة
function getTopSellingProducts($limit = 5) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT p.id, p.name, p.price, 
                   COALESCE(SUM(ti.quantity), 0) AS total_sold,
                   COALESCE(SUM(ti.quantity * (ti.price - ti.discount_amount)), 0) AS total_revenue
            FROM products p
            LEFT JOIN transaction_items ti ON p.id = ti.product_id
            GROUP BY p.id
            ORDER BY total_sold DESC
            LIMIT ?
        ");
        $stmt->bindValue(1, $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching top selling products: " . $e->getMessage());
        return [];
    }
}

// إنشاء فاتورة - النسخة المحسنة
function generateInvoice($transactionId) {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->prepare("
            SELECT t.*, 
                   ti.product_id, ti.name AS item_name, ti.quantity, 
                   ti.price AS unit_price, ti.discount_amount,
                   (ti.price * ti.quantity - ti.discount_amount) AS item_total
            FROM transactions t
            LEFT JOIN transaction_items ti ON t.id = ti.transaction_id
            WHERE t.id = ?
        ");
        $stmt->execute([$transactionId]);
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($items)) {
            return false;
        }
        
        $transaction = [
            'id' => $items[0]['id'],
            'date' => $items[0]['date'],
            'customer_name' => $items[0]['customer_name'] ?? 'زبون عابر',
            'total' => $items[0]['total'],
            'items' => []
        ];
        
        foreach ($items as $item) {
            if ($item['product_id']) {
                $transaction['items'][] = [
                    'id' => $item['product_id'],
                    'name' => $item['item_name'],
                    'quantity' => $item['quantity'],
                    'price' => $item['unit_price'],
                    'discount' => $item['discount_amount'],
                    'total' => $item['item_total']
                ];
            }
        }
        
        return $transaction;
    } catch (PDOException $e) {
        error_log("Error generating invoice: " . $e->getMessage());
        return false;
    }
}

// التحقق من صحة بيانات الزبون
function validateCustomerData($data) {
    $errors = [];
    
    if (empty($data['name'])) {
        $errors[] = 'اسم الزبون مطلوب';
    } elseif (strlen($data['name']) > 100) {
        $errors[] = 'اسم الزبون يجب ألا يتجاوز 100 حرف';
    }
    
    if (isset($data['discounts']) && !is_array($data['discounts'])) {
        $errors[] = 'يجب أن تكون الخصومات مصفوفة';
    }
    
    return $errors;
}

// التحقق من تسجيل الدخول - النسخة المحسنة
function checkLogin() {
    if (!isset($_SESSION['user_id'])) {
        header('Location: login.php');
        exit();
    }
    
    if (!isset($_SESSION['last_activity']) || (time() - $_SESSION['last_activity'] > 3600)) {
        session_unset();
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    $_SESSION['last_activity'] = time();
}