<?php
  
require_once 'config.php';

/**
 * إنشاء اتصال بقاعدة البيانات مع الاحتفاظ به كمتغير ثابت
 */
function getDBConnection() {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4",
                PDO::ATTR_EMULATE_PREPARES => false
            ];
            
            $pdo = new PDO(
                "mysql:host=".DB_HOST.";dbname=".DB_NAME.";charset=utf8mb4", 
                DB_USER, 
                DB_PASS,
                $options
            );
        } catch (PDOException $e) {
            error_log("Connection failed: " . $e->getMessage());
            die("فشل الاتصال بقاعدة البيانات. الرجاء المحاولة لاحقاً.");
        }
    }
    
    return $pdo;
}

/**
 * جلب جميع المنتجات مع عدد مرات بيعها
 */
function getProducts() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT p.*, 
                   (SELECT COUNT(*) FROM transaction_items WHERE product_id = p.id) AS sales_count
            FROM products p 
            ORDER BY p.name ASC
        ");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching products: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب جميع التصنيفات الفريدة للمنتجات
 */
function getCategories() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT DISTINCT category 
            FROM products 
            WHERE category IS NOT NULL AND category != ''
            ORDER BY category ASC
        ");
        $categories = $stmt->fetchAll(PDO::FETCH_COLUMN);
        return $categories ?: ['غير مصنف'];
    } catch (PDOException $e) {
        error_log("Error fetching categories: " . $e->getMessage());
        return ['غير مصنف'];
    }
}

/**
 * جلب جميع الزبائن مع خصوماتهم وعدد طلباتهم
 */
function getCustomers() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT id, name, discounts, credit_balance,
                   (SELECT COUNT(*) FROM transactions WHERE customer_id = customers.id) AS orders_count
            FROM customers 
            ORDER BY name ASC
        ");
        $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($customers as &$customer) {
            $customer['discounts'] = isset($customer['discounts']) && $customer['discounts'] !== null 
                                     ? json_decode($customer['discounts'], true) 
                                     : [];
            $customer['credit_balance'] = (float)$customer['credit_balance'];
        }
        
        return $customers;
    } catch (PDOException $e) {
        error_log("Error fetching customers: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب بيانات زبون معين
 */
function getCustomer($customerId) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE id = ?");
        $stmt->execute([$customerId]);
        $customer = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($customer) {
            $customer['discounts'] = isset($customer['discounts']) && $customer['discounts'] !== null 
                                     ? json_decode($customer['discounts'], true) 
                                     : [];
            $customer['credit_balance'] = (float)$customer['credit_balance'];
        }
        
        return $customer;
    } catch (PDOException $e) {
        error_log("Error fetching customer: " . $e->getMessage());
        return null;
    }
}

/**
 * تحديث خصومات الزبون
 */
function updateCustomerDiscounts($customerId, $discounts) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("UPDATE customers SET discounts = ? WHERE id = ?");
        return $stmt->execute([
            json_encode($discounts, JSON_UNESCAPED_UNICODE),
            $customerId
        ]);
    } catch (PDOException $e) {
        error_log("Error updating customer discounts: " . $e->getMessage());
        return false;
    }
}

/**
 * جلب تاريخ آخر طلب لزبون معين
 */
function getLastOrderDate($customerId) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("SELECT MAX(date) AS last_order_date FROM transactions WHERE customer_id = ?");
        $stmt->execute([$customerId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result && $result['last_order_date']) {
            return date('Y-m-d', strtotime($result['last_order_date']));
        }
        return 'لا يوجد';
    } catch (PDOException $e) {
        error_log("Error fetching last order date for customer ID " . $customerId . ": " . $e->getMessage());
        return 'خطأ';
    }
}


/**
 * جلب أحدث المعاملات
 */
function getRecentTransactions($limit = 5) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                t.id, t.date, t.customer_id, t.customer_name, t.total,
                t.payment_method, t.status, t.notes,
                COUNT(ti.id) AS items_count
            FROM transactions t
            LEFT JOIN customers c ON t.customer_id = c.id
            LEFT JOIN transaction_items ti ON t.id = ti.transaction_id
            GROUP BY t.id, t.date, t.customer_id, t.customer_name, t.total,
                     t.payment_method, t.status, t.notes
            ORDER BY t.date DESC 
            LIMIT ?
        ");
        $stmt->bindValue(1, $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching transactions: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب إحصائيات لوحة التحكم
 */
function getDashboardStats() {
    $pdo = getDBConnection();
    
    $stats = [
        'total_sales' => 0,
        'net_profit' => 0,
        'total_orders' => 0,
        'total_inventory_value' => 0
    ];
    
    try {
        // إجمالي المبيعات
        $stmt = $pdo->query("SELECT COALESCE(SUM(total), 0) AS total_sales FROM transactions");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stats['total_sales'] = (float)$result['total_sales'];
        
        // صافي الربح (مع الأخذ في الاعتبار الخصومات)
        $stmt = $pdo->query("
            SELECT COALESCE(SUM((ti.quantity * ti.price) - ti.discount_amount - (ti.quantity * p.cost)), 0) AS net_profit
            FROM transaction_items ti
            JOIN products p ON ti.product_id = p.id
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stats['net_profit'] = (float)$result['net_profit'];
        
        // إجمالي الطلبات
        $stmt = $pdo->query("SELECT COUNT(*) AS total_orders FROM transactions");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stats['total_orders'] = (int)$result['total_orders'];
        
        // قيمة المخزون (بالتكلفة)
        $stmt = $pdo->query("SELECT COALESCE(SUM(stock * cost), 0) AS total_inventory_value FROM products");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stats['total_inventory_value'] = (float)$result['total_inventory_value'];
        
    } catch (PDOException $e) {
        error_log("Error fetching dashboard stats: " . $e->getMessage());
    }
    
    return $stats;
}

/**
 * إنشاء فاتورة جديدة مع تحديث المخزون
 */
function createTransaction($transactionData) {
    $pdo = getDBConnection();
    try {
        $pdo->beginTransaction();

        // الخطوة 1: التحقق من توفر مخزون كافٍ لجميع المنتجات قبل أي عملية
        foreach ($transactionData['items'] as $item) {
            $stmt = $pdo->prepare("SELECT name, stock FROM products WHERE id = ? FOR UPDATE");
            $stmt->execute([$item['id']]);
            $product = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$product || $product['stock'] < $item['quantity']) {
                throw new PDOException("المخزون غير كافٍ للمنتج: " . ($product['name'] ?? $item['name']));
            }
        }

        // الخطوة 2: إنشاء المعاملة الرئيسية
        $stmt = $pdo->prepare(
            "INSERT INTO transactions (customer_id, customer_name, total, payment_method, status, notes, date) 
             VALUES (?, ?, ?, ?, ?, ?, NOW())"
        );
        $stmt->execute([
            $transactionData['customer_id'],
            $transactionData['customer_name'],
            $transactionData['total'],
            $transactionData['payment_method'],
            $transactionData['status'],
            $transactionData['notes'] ?? ''
        ]);
        $transactionId = $pdo->lastInsertId();

        // الخطوة 3: إضافة بنود المعاملة وتحديث المخزون لكل بند
        foreach ($transactionData['items'] as $item) {
            // إضافة بند المعاملة
            $stmt = $pdo->prepare(
                "INSERT INTO transaction_items (transaction_id, product_id, name, quantity, price, discount_amount) 
                 VALUES (?, ?, ?, ?, ?, ?)"
            );
            $stmt->execute([
                $transactionId, $item['id'], $item['name'], 
                $item['quantity'], $item['price'], $item['discount'] ?? 0
            ]);

            // **الحل**: تحديث كمية المخزون
            $stmt = $pdo->prepare("UPDATE products SET stock = stock - ? WHERE id = ?");
            $stmt->execute([$item['quantity'], $item['id']]);
        }
        
        // الخطوة 4: تحديث رصيد الدين للزبون
        if ($transactionData['payment_method'] === 'credit' && $transactionData['customer_id']) {
            $stmt = $pdo->prepare("UPDATE customers SET credit_balance = credit_balance + ? WHERE id = ?");
            $stmt->execute([$transactionData['total'], $transactionData['customer_id']]);
        }
        
        $pdo->commit();
        return $transactionId;
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Error creating transaction: " . $e->getMessage());
        // إرجاع رسالة الخطأ لتتمكن الواجهة الأمامية من عرضها
        return $e->getMessage(); 
    }
}


/**
 * تسديد دين معاملة
 */
function markTransactionPaid($transactionId) {
    $pdo = getDBConnection();
    try {
        $pdo->beginTransaction();
        
        $stmt = $pdo->prepare("SELECT customer_id, total, status FROM transactions WHERE id = ?");
        $stmt->execute([$transactionId]);
        $transaction = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$transaction || $transaction['status'] === 'paid') {
            $pdo->commit();
            return true; // لا يوجد شيء لفعله إذا كانت مدفوعة بالفعل
        }
        
        $stmt = $pdo->prepare("UPDATE transactions SET status = 'paid' WHERE id = ?");
        $stmt->execute([$transactionId]);
        
        if ($transaction && $transaction['customer_id']) {
            $stmt = $pdo->prepare("UPDATE customers SET credit_balance = credit_balance - ? WHERE id = ?");
            $stmt->execute([$transaction['total'], $transaction['customer_id']]);
        }
        
        $pdo->commit();
        return true;
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Error marking transaction as paid: " . $e->getMessage());
        return false;
    }
}

/**
 * التحقق من تسجيل الدخول
 */
function checkLogin() {
    if (!isset($_SESSION['user_id'])) {
        header('Location: login.php');
        exit();
    }
    
    if (!isset($_SESSION['last_activity']) || (time() - $_SESSION['last_activity'] > 3600)) {
        session_unset();
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    $_SESSION['last_activity'] = time();
}

/**
 * جلب المنتجات الأكثر مبيعاً
 */
function getTopSellingProducts($limit = 5) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT p.id, p.name, p.price, 
                   COALESCE(SUM(ti.quantity), 0) AS total_sold,
                   COALESCE(SUM(ti.quantity * ti.price), 0) AS total_revenue
            FROM products p
            LEFT JOIN transaction_items ti ON p.id = ti.product_id
            GROUP BY p.id
            ORDER BY total_sold DESC
            LIMIT ?
        ");
        $stmt->bindValue(1, $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching top selling products: " . $e->getMessage());
        return [];
    }
}

/**
 * إنشاء فاتورة PDF
 */
function generateInvoice($transactionId) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT t.*, 
                   ti.product_id, ti.name AS item_name, ti.quantity, 
                   ti.price AS unit_price, ti.discount_amount,
                   (ti.price * ti.quantity - ti.discount_amount) AS item_total
            FROM transactions t
            LEFT JOIN transaction_items ti ON t.id = ti.transaction_id
            WHERE t.id = ?
        ");
        $stmt->execute([$transactionId]);
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($items)) {
            return false;
        }
        
        $transaction = [
            'id' => $items[0]['id'],
            'date' => $items[0]['date'],
            'customer_name' => $items[0]['customer_name'] ?? 'زبون عابر',
            'total' => $items[0]['total'],
            'items' => []
        ];
        
        foreach ($items as $item) {
            if ($item['product_id']) {
                $transaction['items'][] = [
                    'id' => $item['product_id'],
                    'name' => $item['item_name'],
                    'quantity' => $item['quantity'],
                    'price' => $item['unit_price'],
                    'discount' => $item['discount_amount'],
                    'total' => $item['item_total']
                ];
            }
        }
        
        return $transaction;
    } catch (PDOException $e) {
        error_log("Error generating invoice: " . $e->getMessage());
        return false;
    }
}

/**
 * جلب جميع الطلبيات (المعاملات)
 */
function getTransactions() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT t.id, t.date, c.name AS customer_name, t.total, t.payment_method, t.status
            FROM transactions t
            LEFT JOIN customers c ON t.customer_id = c.id
            ORDER BY t.date DESC
        ");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching transactions: " . $e->getMessage());
        return [];
    }
}

/**
 * حذف طلبية مع بنودها وإعادة المخزون وتعديل رصيد الزبون
 */
function deleteTransaction($transactionId) {
    $pdo = getDBConnection();
    try {
        $pdo->beginTransaction();

        // جلب المعاملة لتعديل رصيد الزبون إذا كانت بالدين
        $stmtTransaction = $pdo->prepare("SELECT customer_id, total, payment_method, status FROM transactions WHERE id = ?");
        $stmtTransaction->execute([$transactionId]);
        $transaction = $stmtTransaction->fetch(PDO::FETCH_ASSOC);

        // جلب بنود المعاملة لاستعادة المخزون
        $stmtItems = $pdo->prepare("SELECT product_id, quantity FROM transaction_items WHERE transaction_id = ?");
        $stmtItems->execute([$transactionId]);
        $items = $stmtItems->fetchAll(PDO::FETCH_ASSOC);

        // استعادة المخزون لكل منتج
        foreach ($items as $item) {
            if ($item['product_id']) {
                $stmtUpdateStock = $pdo->prepare("UPDATE products SET stock = stock + ? WHERE id = ?");
                $stmtUpdateStock->execute([$item['quantity'], $item['product_id']]);
            }
        }
        
        // إذا كانت المعاملة بالدين أو غير مدفوعة، يتم خصم قيمتها من رصيد الزبون
        if ($transaction && $transaction['customer_id'] && ($transaction['payment_method'] === 'credit' || $transaction['status'] === 'unpaid')) {
            $stmtUpdateCredit = $pdo->prepare("UPDATE customers SET credit_balance = credit_balance - ? WHERE id = ?");
            $stmtUpdateCredit->execute([$transaction['total'], $transaction['customer_id']]);
        }

        // حذف بنود المعاملة
        $stmtDeleteItems = $pdo->prepare("DELETE FROM transaction_items WHERE transaction_id = ?");
        $stmtDeleteItems->execute([$transactionId]);

        // حذف المعاملة نفسها
        $stmtDeleteTrans = $pdo->prepare("DELETE FROM transactions WHERE id = ?");
        $success = $stmtDeleteTrans->execute([$transactionId]);

        $pdo->commit();
        return $success;
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Error deleting transaction: " . $e->getMessage());
        return false;
    }
}


/**
 * ==============================================
 * دوال التقارير الجديدة
 * ==============================================
 */

/**
 * حساب نطاق التاريخ للتقرير
 */
function calculateReportDateRange($period, $customStart = '', $customEnd = '') {
    $today = date('Y-m-d');
    $result = ['start' => $today, 'end' => $today];
    
    switch ($period) {
        case 'today':
            break;
        case 'yesterday':
            $result['start'] = $result['end'] = date('Y-m-d', strtotime('-1 day'));
            break;
        case 'this_week':
            $result['start'] = date('Y-m-d', strtotime('monday this week'));
            $result['end'] = date('Y-m-d', strtotime('sunday this week'));
            break;
        case 'this_month':
            $result['start'] = date('Y-m-01');
            $result['end'] = date('Y-m-t');
            break;
        case 'last_month':
            $result['start'] = date('Y-m-01', strtotime('first day of last month'));
            $result['end'] = date('Y-m-t', strtotime('last day of last month'));
            break;
        case 'custom':
            if (!empty($customStart) && !empty($customEnd)) {
                $result['start'] = $customStart;
                $result['end'] = $customEnd;
            }
            break;
    }
    
    return $result;
}

/**
 * جلب بيانات تقرير المبيعات اليومية
 */
function getDailySalesReport($startDate, $endDate) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                DATE(t.date) AS transaction_date,
                COUNT(t.id) AS daily_orders_count,
                SUM(t.total) AS daily_sales_total
            FROM transactions t
            WHERE t.date BETWEEN ? AND ?
            AND t.status = 'paid'
            GROUP BY DATE(t.date)
            ORDER BY DATE(t.date) DESC
        ");
        $stmt->execute([$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching daily sales report: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب تقرير المخزون الحالي
 */
function getCurrentInventoryReport() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT 
                p.id,
                p.name AS product_name,
                p.stock AS stock_quantity,
                p.cost,
                p.price AS unit_price,
                p.category,
                'غير محدد' AS supplier
            FROM products p
            ORDER BY p.stock ASC
        ");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching inventory report: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب تقرير الديون للعملاء
 */
function getCustomerDebtsReport($startDate, $endDate) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                c.id,
                c.name,
                c.credit_balance,
                (SELECT COUNT(*) FROM transactions t 
                 WHERE t.customer_id = c.id 
                 AND t.status = 'unpaid'
                 AND t.date BETWEEN ? AND ?) AS unpaid_orders
            FROM customers c
            WHERE c.credit_balance > 0
            ORDER BY c.credit_balance DESC
        ");
        $stmt->execute([$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching customer debts report: " . $e->getMessage());
        return [];
    }
}

/**
 * حساب القيمة الإجمالية للمخزون (حسب تكلفة الشراء)
 */
function getTotalInventoryValue() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT SUM(cost * stock) AS total_value 
            FROM products
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return (float)($result['total_value'] ?? 0);
    } catch (PDOException $e) {
        error_log("Error calculating inventory value: " . $e->getMessage());
        return 0;
    }
}

/**
 * حساب إجمالي الربح لفترة محددة (مع الأخذ في الاعتبار الخصومات)
 */
function calculateTotalProfit($startDate, $endDate) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                COALESCE(SUM((ti.quantity * ti.price) - ti.discount_amount - (ti.quantity * p.cost)), 0) AS total_profit
            FROM transactions t
            JOIN transaction_items ti ON t.id = ti.transaction_id
            JOIN products p ON ti.product_id = p.id
            WHERE t.date BETWEEN ? AND ?
            AND t.status = 'paid'
        ");
        $stmt->execute([$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return (float)($result['total_profit'] ?? 0);
    } catch (PDOException $e) {
        error_log("Error calculating total profit: " . $e->getMessage());
        return 0;
    }
}