<?php
require_once 'config.php';

// تمكين عرض الأخطاء للتصحيح
error_reporting(E_ALL);
ini_set('display_errors', 1);

// معالجة الطلبات الواردة
if (isset($_GET['action'])) {
    header('Content-Type: application/json');
    session_start();
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if ($_GET['action'] === 'create_transaction') {
            if (!isset($_SESSION['user_id'])) {
                throw new Exception("غير مسموح بالوصول: يلزم تسجيل الدخول");
            }
            
            $result = createTransaction($input);
            echo json_encode($result);
            exit();
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'خطأ في معالجة الطلب: ' . $e->getMessage()
        ]);
        exit();
    }
}

/**
 * إنشاء اتصال بقاعدة البيانات
 */
function getDBConnection() {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4",
                PDO::ATTR_EMULATE_PREPARES => false
            ];
            
            $pdo = new PDO(
                "mysql:host=".DB_HOST.";dbname=".DB_NAME.";charset=utf8mb4", 
                DB_USER, 
                DB_PASS,
                $options
            );
        } catch (PDOException $e) {
            error_log("Connection failed: " . $e->getMessage());
            die("فشل الاتصال بقاعدة البيانات. الرجاء المحاولة لاحقاً.");
        }
    }
    
    return $pdo;
}

/**
 * إنشاء فاتورة جديدة
 */
function createTransaction($transactionData) {
    $pdo = getDBConnection();
    
    try {
        // التحقق من صحة البيانات الأساسية
        if (!isset($transactionData['total'], $transactionData['payment_method'])) {
            throw new Exception("بيانات المعاملة غير مكتملة");
        }

        $pdo->beginTransaction();

        // إدخال المعاملة الأساسية
        $stmt = $pdo->prepare("
            INSERT INTO transactions 
            (customer_id, customer_name, total, payment_method, status, notes, date) 
            VALUES (:customer_id, :customer_name, :total, :payment_method, :status, :notes, NOW())
        ");
        
        $stmt->execute([
            ':customer_id' => $transactionData['customer_id'] ?? null,
            ':customer_name' => $transactionData['customer_name'] ?? 'زبون عابر',
            ':total' => $transactionData['total'],
            ':payment_method' => $transactionData['payment_method'],
            ':status' => $transactionData['status'] ?? ($transactionData['payment_method'] === 'cash' ? 'paid' : 'unpaid'),
            ':notes' => $transactionData['notes'] ?? ''
        ]);
        
        $transactionId = $pdo->lastInsertId();

        // معالجة العناصر
        if (empty($transactionData['items']) || !is_array($transactionData['items'])) {
            throw new Exception("لا توجد عناصر في الفاتورة");
        }

        foreach ($transactionData['items'] as $item) {
            if (empty($item['id']) || empty($item['quantity']) || empty($item['price'])) {
                continue;
            }

            // إدخال عنصر الفاتورة
            $stmt = $pdo->prepare("
                INSERT INTO transaction_items 
                (transaction_id, product_id, name, quantity, price, discount_amount) 
                VALUES (:transaction_id, :product_id, :name, :quantity, :price, :discount)
            ");
            
            $stmt->execute([
                ':transaction_id' => $transactionId,
                ':product_id' => $item['id'],
                ':name' => $item['name'] ?? 'منتج غير معروف',
                ':quantity' => $item['quantity'],
                ':price' => $item['price'],
                ':discount' => $item['discount'] ?? 0
            ]);

            // تحديث المخزون
            $stmt = $pdo->prepare("UPDATE products SET stock = stock - :quantity WHERE id = :product_id");
            $stmt->execute([
                ':quantity' => $item['quantity'],
                ':product_id' => $item['id']
            ]);
        }

        // معالجة الديون إذا كانت العملية دين
        if ($transactionData['payment_method'] === 'credit' && !empty($transactionData['customer_id'])) {
            $stmt = $pdo->prepare("UPDATE customers SET credit_balance = credit_balance + :total WHERE id = :customer_id");
            $stmt->execute([
                ':total' => $transactionData['total'],
                ':customer_id' => $transactionData['customer_id']
            ]);
        }

        $pdo->commit();
        
        return [
            'success' => true,
            'transaction_id' => $transactionId,
            'message' => 'تمت العملية بنجاح'
        ];
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Error in createTransaction: " . $e->getMessage());
        return [
            'success' => false,
            'message' => 'حدث خطأ في قاعدة البيانات: ' . $e->getMessage()
        ];
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Error in createTransaction: " . $e->getMessage());
        return [
            'success' => false,
            'message' => $e->getMessage()
        ];
    }
}

// الدوال المساعدة للطلبيات
function getOrders() {
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM orders ORDER BY created_at DESC");
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function getOrderStats() {
    global $conn;
    $stats = ['pending' => 0, 'processing' => 0, 'completed' => 0, 'cancelled' => 0];
    
    $stmt = $conn->prepare("SELECT status, COUNT(*) as count FROM orders GROUP BY status");
    $stmt->execute();
    
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $stats[$row['status']] = $row['count'];
    }
    
    return $stats;
}

function getOrderDetails($order_id) {
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM orders WHERE id = ?");
    $stmt->execute([$order_id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function getOrderItems($order_id) {
    global $conn;
    $stmt = $conn->prepare("
        SELECT oi.*, p.name 
        FROM order_items oi 
        JOIN products p ON oi.product_id = p.id 
        WHERE oi.order_id = ?
    ");
    $stmt->execute([$order_id]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function deleteOrder($order_id) {
    global $conn;
    
    try {
        $conn->beginTransaction();
        
        // حذف عناصر الطلب أولاً
        $stmt = $conn->prepare("DELETE FROM order_items WHERE order_id = ?");
        $stmt->execute([$order_id]);
        
        // ثم حذف الطلب نفسه
        $stmt = $conn->prepare("DELETE FROM orders WHERE id = ?");
        $stmt->execute([$order_id]);
        
        $conn->commit();
        return true;
    } catch (Exception $e) {
        $conn->rollBack();
        return false;
    }
}

function getOrderStatusText($status) {
    $statuses = [
        'pending' => 'جديد',
        'processing' => 'قيد التجهيز',
        'completed' => 'مكتمل',
        'cancelled' => 'ملغى'
    ];
    return $statuses[$status] ?? $status;
}

function formatOrderAddress($order) {
    return "
        <p><strong>الاسم:</strong> {$order['customer_name']}</p>
        <p><strong>الهاتف:</strong> {$order['customer_phone']}</p>
        <p><strong>العنوان:</strong> {$order['customer_address']}</p>
        <p><strong>المدينة:</strong> {$order['customer_city']}</p>
    ";
}

/**
 * جلب أحدث المعاملات
 */
function getRecentTransactions($limit = 5) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                t.id, 
                t.date, 
                t.customer_name, 
                t.total,
                t.payment_method, 
                t.status,
                COUNT(ti.id) AS items_count
            FROM transactions t
            LEFT JOIN transaction_items ti ON t.id = ti.transaction_id
            GROUP BY t.id
            ORDER BY t.date DESC 
            LIMIT :limit
        ");
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching transactions: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب إحصائيات لوحة التحكم
 */
function getDashboardStats() {
    $pdo = getDBConnection();
    
    $stats = [
        'total_sales' => 0,
        'net_profit' => 0,
        'total_orders' => 0,
        'active_customers' => 0
    ];
    
    try {
        // إجمالي المبيعات
        $stmt = $pdo->query("SELECT COALESCE(SUM(total), 0) AS total_sales FROM transactions");
        $stats['total_sales'] = (float)$stmt->fetchColumn();

        // إجمالي الربح الصافي
        try {
            $stmt = $pdo->query("SELECT COALESCE(SUM(total - total_cost), 0) AS net_profit FROM transactions");
            $stats['net_profit'] = (float)$stmt->fetchColumn();
        } catch (PDOException $e) {
            $stats['net_profit'] = 0;
        }

        // إجمالي عدد الطلبات
        $stmt = $pdo->query("SELECT COUNT(*) AS total_orders FROM transactions");
        $stats['total_orders'] = (int)$stmt->fetchColumn();

        // عدد الزبائن النشطين
        $stmt = $pdo->query("SELECT COUNT(DISTINCT customer_id) AS active_customers FROM transactions WHERE customer_id IS NOT NULL");
        $stats['active_customers'] = (int)$stmt->fetchColumn();

    } catch (PDOException $e) {
        error_log("Error fetching dashboard stats: " . $e->getMessage());
    }
    
    return $stats;
}



/**
 * جلب جميع المنتجات مع عدد مرات بيعها
 */
function getProducts() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT p.*, 
                   (SELECT COUNT(*) FROM transaction_items WHERE product_id = p.id) AS sales_count
            FROM products p 
            ORDER BY p.name ASC
        ");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching products: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب جميع التصنيفات الفريدة للمنتجات
 */
function getCategories() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT DISTINCT category 
            FROM products 
            WHERE category IS NOT NULL AND category != ''
            ORDER BY category ASC
        ");
        $categories = $stmt->fetchAll(PDO::FETCH_COLUMN);
        return $categories ?: ['غير مصنف'];
    } catch (PDOException $e) {
        error_log("Error fetching categories: " . $e->getMessage());
        return ['غير مصنف'];
    }
}

/**
 * جلب جميع الزبائن مع خصوماتهم وعدد طلباتهم
 */
function getCustomers() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT 
                id, 
                name, 
                discounts, 
                credit_balance,
                (SELECT COUNT(*) FROM transactions WHERE customer_id = customers.id) AS orders_count
            FROM customers 
            ORDER BY name ASC
        ");
        $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($customers as &$customer) {
            $customer['discounts'] = json_decode($customer['discounts'], true) ?? [];
            $customer['credit_balance'] = (float)$customer['credit_balance'];
        }
        
        return $customers;
    } catch (PDOException $e) {
        error_log("Error fetching customers: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب بيانات زبون معين
 */
function getCustomer($customerId) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE id = :id");
        $stmt->bindParam(':id', $customerId, PDO::PARAM_INT);
        $stmt->execute();
        $customer = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($customer) {
            $customer['discounts'] = json_decode($customer['discounts'], true) ?? [];
            $customer['credit_balance'] = (float)$customer['credit_balance'];
        }
        
        return $customer;
    } catch (PDOException $e) {
        error_log("Error fetching customer: " . $e->getMessage());
        return null;
    }
}

/**
 * تحديث خصومات الزبون
 */
function updateCustomerDiscounts($customerId, $discounts) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("UPDATE customers SET discounts = :discounts WHERE id = :id");
        $stmt->bindParam(':id', $customerId, PDO::PARAM_INT);
        $stmt->bindValue(':discounts', json_encode($discounts, JSON_UNESCAPED_UNICODE));
        return $stmt->execute();
    } catch (PDOException $e) {
        error_log("Error updating customer discounts: " . $e->getMessage());
        return false;
    }
}

/**
 * جلب تاريخ آخر طلب لزبون معين
 */
function getLastOrderDate($customerId) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("SELECT MAX(date) AS last_order_date FROM transactions WHERE customer_id = :customerId");
        $stmt->bindParam(':customerId', $customerId, PDO::PARAM_INT);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result && $result['last_order_date']) {
            return date('Y-m-d', strtotime($result['last_order_date']));
        }
        return 'لا يوجد';
    } catch (PDOException $e) {
        error_log("Error fetching last order date: " . $e->getMessage());
        return 'خطأ';
    }
}

/**
 * تسديد دين معاملة
 */
function markTransactionPaid($transactionId) {
    $pdo = getDBConnection();
    try {
        $pdo->beginTransaction();
        
        $stmt = $pdo->prepare("UPDATE transactions SET status = 'paid' WHERE id = :id");
        $stmt->bindParam(':id', $transactionId, PDO::PARAM_INT);
        $stmt->execute();
        
        $stmt = $pdo->prepare("SELECT customer_id, total FROM transactions WHERE id = :id");
        $stmt->bindParam(':id', $transactionId, PDO::PARAM_INT);
        $stmt->execute();
        $transaction = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($transaction && $transaction['customer_id']) {
            $stmt = $pdo->prepare("UPDATE customers SET credit_balance = credit_balance - :total WHERE id = :customer_id");
            $stmt->bindParam(':total', $transaction['total']);
            $stmt->bindParam(':customer_id', $transaction['customer_id'], PDO::PARAM_INT);
            $stmt->execute();
        }
        
        $pdo->commit();
        return true;
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Error marking transaction as paid: " . $e->getMessage());
        return false;
    }
}

/**
 * التحقق من تسجيل الدخول
 */
function checkLogin() {
    if (!isset($_SESSION['user_id'])) {
        header('Location: login.php');
        exit();
    }
    
    if (!isset($_SESSION['last_activity']) || (time() - $_SESSION['last_activity'] > 3600)) {
        session_unset();
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    $_SESSION['last_activity'] = time();
}

/**
 * جلب المنتجات الأكثر مبيعاً
 */
function getTopSellingProducts($limit = 5) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                p.id, 
                p.name, 
                p.price, 
                COALESCE(SUM(ti.quantity), 0) AS total_sold,
                COALESCE(SUM(ti.quantity * ti.price), 0) AS total_revenue
            FROM products p
            LEFT JOIN transaction_items ti ON p.id = ti.product_id
            GROUP BY p.id, p.name, p.price
            ORDER BY total_sold DESC
            LIMIT :limit
        ");
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching top selling products: " . $e->getMessage());
        return [];
    }
}

/**
 * إنشاء فاتورة PDF
 */
function generateInvoice($transactionId) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                t.*, 
                ti.product_id, 
                ti.name AS item_name, 
                ti.quantity, 
                ti.price AS unit_price, 
                ti.discount_amount,
                (ti.price * ti.quantity - ti.discount_amount) AS item_total
            FROM transactions t
            LEFT JOIN transaction_items ti ON t.id = ti.transaction_id
            WHERE t.id = :id
        ");
        $stmt->bindParam(':id', $transactionId, PDO::PARAM_INT);
        $stmt->execute();
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($items)) {
            return false;
        }
        
        $transaction = [
            'id' => $items[0]['id'],
            'date' => $items[0]['date'],
            'customer_name' => $items[0]['customer_name'] ?? 'زبون عابر',
            'total' => $items[0]['total'],
            'items' => []
        ];
        
        foreach ($items as $item) {
            if ($item['product_id']) {
                $transaction['items'][] = [
                    'id' => $item['product_id'],
                    'name' => $item['item_name'],
                    'quantity' => $item['quantity'],
                    'price' => $item['unit_price'],
                    'discount' => $item['discount_amount'],
                    'total' => $item['item_total']
                ];
            }
        }
        
        return $transaction;
    } catch (PDOException $e) {
        error_log("Error generating invoice: " . $e->getMessage());
        return false;
    }
}

/**
 * حساب نطاق التواريخ بناءً على الفترة الزمنية المحددة
 */
function calculateDateRange($timePeriod, $startDate = null, $endDate = null) {
    $start = '';
    $end = '';

    switch ($timePeriod) {
        case 'today':
            $start = date('Y-m-d');
            $end = date('Y-m-d');
            break;
        case 'yesterday':
            $start = date('Y-m-d', strtotime('-1 day'));
            $end = date('Y-m-d', strtotime('-1 day'));
            break;
        case 'this_week':
            $start = date('Y-m-d', strtotime('monday this week'));
            $end = date('Y-m-d', strtotime('sunday this week'));
            break;
        case 'last_week':
            $start = date('Y-m-d', strtotime('monday last week'));
            $end = date('Y-m-d', strtotime('sunday last week'));
            break;
        case 'this_month':
            $start = date('Y-m-d', strtotime('first day of this month'));
            $end = date('Y-m-d', strtotime('last day of this month'));
            break;
        case 'last_month':
            $start = date('Y-m-d', strtotime('first day of last month'));
            $end = date('Y-m-d', strtotime('last day of last month'));
            break;
        case 'this_year':
            $start = date('Y-m-d', strtotime('first day of January this year'));
            $end = date('Y-m-d', strtotime('last day of December this year'));
            break;
        case 'custom':
            $start = $startDate;
            $end = $endDate;
            break;
        default:
            $start = date('Y-m-d', strtotime('first day of this month'));
            $end = date('Y-m-d', strtotime('last day of this month'));
            break;
    }

    return ['start' => $start, 'end' => $end];
}

/**
 * جلب تقرير المبيعات
 */
function getSalesReport($startDate, $endDate) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                DATE(t.date) as transaction_date,
                SUM(t.total) as daily_sales_total,
                COUNT(t.id) as daily_orders_count
            FROM transactions t
            WHERE DATE(t.date) BETWEEN :startDate AND :endDate
            GROUP BY transaction_date
            ORDER BY transaction_date ASC
        ");
        $stmt->bindParam(':startDate', $startDate);
        $stmt->bindParam(':endDate', $endDate);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching sales report: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب تقرير الزبائن
 */
function getCustomersReport($startDate, $endDate) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                c.id,
                c.name as customer_name,
                c.credit_balance,
                COUNT(t.id) as total_orders,
                SUM(t.total) as total_spent
            FROM customers c
            LEFT JOIN transactions t ON c.id = t.customer_id
            WHERE (DATE(t.date) BETWEEN :startDate AND :endDate) OR t.id IS NULL
            GROUP BY c.id, c.name, c.credit_balance
            ORDER BY total_spent DESC
        ");
        $stmt->bindParam(':startDate', $startDate);
        $stmt->bindParam(':endDate', $endDate);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching customers report: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب تقرير المنتجات
 */
function getProductsReport($startDate, $endDate) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                p.id,
                p.name as product_name,
                p.stock_quantity,
                p.price as unit_price,
                p.supplier,
                p.category,
                COALESCE(SUM(ti.quantity), 0) as total_quantity_sold,
                COALESCE(SUM(ti.quantity * ti.price), 0) as total_revenue
            FROM products p
            LEFT JOIN transaction_items ti ON p.id = ti.product_id
            LEFT JOIN transactions t ON ti.transaction_id = t.id
            WHERE DATE(t.date) BETWEEN :startDate AND :endDate
            GROUP BY p.id, p.name, p.stock_quantity, p.price, p.supplier, p.category
            ORDER BY total_quantity_sold DESC
        ");
        $stmt->bindParam(':startDate', $startDate);
        $stmt->bindParam(':endDate', $endDate);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching products report: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب تقرير المعاملات
 */
function getTransactionsReport($startDate, $endDate) {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->prepare("
            SELECT 
                t.id, 
                t.date, 
                t.customer_name, 
                t.total, 
                t.payment_method, 
                t.status,
                GROUP_CONCAT(CONCAT(ti.name, ' (', ti.quantity, ')') SEPARATOR ', ') as items_summary
            FROM transactions t
            LEFT JOIN transaction_items ti ON t.id = ti.transaction_id
            WHERE DATE(t.date) BETWEEN :startDate AND :endDate
            GROUP BY t.id, t.date, t.customer_name, t.total, t.payment_method, t.status
            ORDER BY t.date DESC
        ");
        $stmt->bindParam(':startDate', $startDate);
        $stmt->bindParam(':endDate', $endDate);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching transactions report: " . $e->getMessage());
        return [];
    }
}

/**
 * جلب تقرير المخزون
 */
function getInventoryReport() {
    $pdo = getDBConnection();
    try {
        $stmt = $pdo->query("
            SELECT 
                p.id,
                p.name as product_name,
                p.stock_quantity,
                p.price as unit_price,
                p.supplier,
                p.category,
                (SELECT COUNT(*) FROM transaction_items WHERE product_id = p.id) as times_sold
            FROM products p
            ORDER BY p.name ASC
        ");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error fetching inventory report: " . $e->getMessage());
        return [];
    }
}