<?php
session_start();
require_once 'config.php';
require_once 'functions.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// جلب جميع الديون غير المسددة
$pdo = getDBConnection();
$stmt = $pdo->prepare("
    SELECT t.*, c.name AS customer_name, c.credit_balance 
    FROM transactions t
    LEFT JOIN customers c ON t.customer_id = c.id
    WHERE t.status = 'unpaid' AND t.payment_method = 'credit'
    ORDER BY t.date DESC
");
$stmt->execute();
$debts = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة الديون - نظام المبيعات</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .debt-row:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }
        .unpaid-badge {
            animation: pulse 2s infinite;
        }
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.7; }
            100% { opacity: 1; }
        }
    </style>
</head>
<body class="bg-gray-50 font-sans">
    <nav class="bg-blue-600 text-white shadow-lg">
        <div class="container mx-auto px-4 py-3 flex justify-between items-center">
            <h1 class="text-xl font-bold">
                <i class="fas fa-money-bill-wave mr-2"></i> إدارة الديون
            </h1>
            <div>
                <a href="pos.php" class="px-3 py-2 rounded hover:bg-blue-700 transition">
                    <i class="fas fa-cash-register mr-1"></i> نقطة البيع
                </a>
                <a href="customers.php" class="px-3 py-2 rounded hover:bg-blue-700 transition">
                    <i class="fas fa-users mr-1"></i> العملاء
                </a>
                <a href="logout.php" class="px-3 py-2 rounded hover:bg-red-600 transition">
                    <i class="fas fa-sign-out-alt mr-1"></i> تسجيل الخروج
                </a>
            </div>
        </div>
    </nav>

    <div class="container mx-auto px-4 py-6">
        <div class="flex justify-between items-center mb-6">
            <h2 class="text-2xl font-bold text-gray-800">
                <i class="fas fa-file-invoice-dollar mr-2 text-blue-600"></i> قائمة الديون غير المسددة
            </h2>
            <div class="bg-blue-100 text-blue-800 px-4 py-2 rounded-lg">
                <i class="fas fa-info-circle mr-1"></i> إجمالي الديون: 
                <span class="font-bold"><?= number_format(array_sum(array_column($debts, 'total')), 2) ?> د.م</span>
            </div>
        </div>

        <div class="bg-white rounded-xl shadow-md overflow-hidden">
            <div class="px-6 py-4 bg-gray-50 border-b flex justify-between items-center">
                <div class="relative w-64">
                    <input type="text" id="searchDebts" placeholder="بحث في الديون..." class="w-full pl-10 pr-4 py-2 rounded-lg border border-gray-300 focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <i class="fas fa-search absolute left-3 top-3 text-gray-400"></i>
                </div>
            </div>

            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead class="bg-gray-100 text-gray-600">
                        <tr>
                            <th class="px-6 py-3 text-right font-semibold">#</th>
                            <th class="px-6 py-3 text-right font-semibold">الزبون</th>
                            <th class="px-6 py-3 text-right font-semibold">التاريخ</th>
                            <th class="px-6 py-3 text-right font-semibold">المبلغ</th>
                            <th class="px-6 py-3 text-right font-semibold">الرصيد</th>
                            <th class="px-6 py-3 text-right font-semibold">الحالة</th>
                            <th class="px-6 py-3 text-right font-semibold">إجراءات</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-200">
                        <?php if (empty($debts)): ?>
                            <tr>
                                <td colspan="7" class="px-6 py-4 text-center text-gray-500">
                                    <i class="fas fa-check-circle text-green-500 text-2xl mb-2"></i>
                                    <p class="text-lg">لا توجد ديون غير مسددة</p>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($debts as $debt): ?>
                            <tr class="bg-white hover:bg-gray-50 transition debt-row">
                                <td class="px-6 py-4 font-medium text-gray-900"><?= $debt['id'] ?></td>
                                <td class="px-6 py-4">
                                    <div class="flex items-center">
                                        <div class="flex-shrink-0 h-10 w-10 bg-blue-100 rounded-full flex items-center justify-center">
                                            <i class="fas fa-user text-blue-600"></i>
                                        </div>
                                        <div class="mr-3">
                                            <p class="text-gray-900 font-medium"><?= htmlspecialchars($debt['customer_name'] ?? 'زبون عابر') ?></p>
                                            <p class="text-gray-500 text-sm"><?= $debt['customer_id'] ? 'دائم' : 'عابر' ?></p>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 text-gray-700"><?= date('Y/m/d H:i', strtotime($debt['date'])) ?></td>
                                <td class="px-6 py-4 font-bold text-red-600"><?= number_format($debt['total'], 2) ?> د.م</td>
                                <td class="px-6 py-4 font-medium <?= $debt['credit_balance'] < 0 ? 'text-red-600' : 'text-green-600' ?>">
                                    <?= number_format($debt['credit_balance'], 2) ?> د.م
                                </td>
                                <td class="px-6 py-4">
                                    <span class="px-3 py-1 rounded-full text-xs font-semibold bg-red-100 text-red-800 unpaid-badge">
                                        <i class="fas fa-clock mr-1"></i> غير مسدد
                                    </span>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="flex space-x-2">
                                        <button onclick="markAsPaid(<?= $debt['id'] ?>)" 
                                                class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition flex items-center">
                                            <i class="fas fa-check-circle mr-2"></i> تسديد
                                        </button>
                                        <a href="invoice.php?id=<?= $debt['id'] ?>" 
                                           class="bg-blue-100 text-blue-600 px-4 py-2 rounded-lg hover:bg-blue-200 transition flex items-center">
                                            <i class="fas fa-eye mr-2"></i> التفاصيل
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <div class="px-6 py-4 bg-gray-50 border-t flex justify-between items-center">
                <div class="text-gray-500">
                    عرض <span class="font-medium">1</span> إلى <span class="font-medium"><?= count($debts) ?></span> من <span class="font-medium"><?= count($debts) ?></span> سجلات
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
    function markAsPaid(transactionId) {
        if (confirm('هل أنت متأكد من تسديد هذا الدين؟')) {
            const btn = $(`button[onclick="markAsPaid(${transactionId})"]`);
            btn.html('<i class="fas fa-spinner fa-spin mr-2"></i> جاري المعالجة');
            btn.prop('disabled', true);

            $.ajax({
                url: 'api/transactions.php',
                method: 'POST',
                data: { 
                    id: transactionId,
                    action: 'mark_paid'
                },
                success: function(response) {
                    if (response.success) {
                        showAlert('success', 'تم تسديد الدين بنجاح');
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        showAlert('error', response.message || 'فشل في تسديد الدين');
                        resetButton(btn);
                    }
                },
                error: function(xhr) {
                    showAlert('error', 'حدث خطأ في الاتصال: ' + xhr.statusText);
                    resetButton(btn);
                }
            });
        }
    }

    function resetButton(btn) {
        btn.html('<i class="fas fa-check-circle mr-2"></i> تسديد');
        btn.prop('disabled', false);
    }

    function showAlert(type, message) {
        const alert = $(`
            <div class="fixed top-4 right-4 z-50 px-6 py-4 rounded-lg shadow-lg text-white flex items-center ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-2"></i>
                ${message}
            </div>
        `);
        
        $('body').append(alert);
        setTimeout(() => alert.fadeOut(500, () => alert.remove()), 3000);
    }

    // بحث سريع في الجدول
    $('#searchDebts').on('input', function() {
        const searchTerm = $(this).val().toLowerCase();
        $('.debt-row').each(function() {
            const rowText = $(this).text().toLowerCase();
            $(this).toggle(rowText.includes(searchTerm));
        });
    });
    </script>
</body>
</html>