<?php
session_start();
require_once 'config.php';
require_once 'functions.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// جلب جميع الزبائن
$customers = getCustomers();
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة الزبائن - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body { font-family: 'Tajawal', sans-serif; }
        .customer-card:hover { transform: translateY(-3px); box-shadow: 0 10px 20px rgba(0,0,0,0.1); }
        [type="checkbox"]:checked { background-color: #3b82f6; }
    </style>
</head>
<body class="bg-gray-100">
    <div class="flex h-screen overflow-hidden">
        <?php include 'sidebar.php'; ?>

        <div class="flex-1 overflow-auto">
            <header class="bg-white shadow-sm p-4 flex justify-between items-center">
                <h2 class="text-xl font-bold text-gray-800">إدارة الزبائن</h2>
                <div class="flex items-center space-x-4">
                    <button id="addCustomerBtn" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700">
                        <i class="fas fa-plus mr-2"></i> إضافة زبون جديد
                    </button>
                    <div class="flex items-center">
                        <span class="mr-2">مرحباً, <?php echo $_SESSION['username']; ?></span>
                        <img src="https://ui-avatars.com/api/?name=<?php echo urlencode($_SESSION['username']); ?>&background=random" class="w-8 h-8 rounded-full">
                    </div>
                </div>
            </header>

            <main class="p-4">
                <!-- فلترة الزبائن -->
                <div class="bg-white rounded-lg shadow p-4 mb-4">
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">بحث بالاسم</label>
                            <input type="text" id="searchByName" placeholder="ابحث باسم الزبون..." class="w-full p-2 border rounded-lg">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">فلترة حسب الرصيد</label>
                            <select id="filterByCredit" class="w-full p-2 border rounded-lg">
                                <option value="all">الكل</option>
                                <option value="positive">رصيد موجب</option>
                                <option value="negative">رصيد مدين</option>
                                <option value="zero">بدون رصيد</option>
                            </select>
                        </div>
                        <div class="flex items-end">
                            <button id="resetFilters" class="px-4 py-2 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300 w-full">
                                <i class="fas fa-undo mr-2"></i> إعادة تعيين
                            </button>
                        </div>
                    </div>
                </div>

                <!-- إحصائيات سريعة -->
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                    <div class="bg-white rounded-lg shadow p-4">
                        <div class="flex items-center">
                            <div class="p-3 rounded-full bg-blue-100 text-blue-600 mr-3">
                                <i class="fas fa-users text-xl"></i>
                            </div>
                            <div>
                                <p class="text-gray-500">إجمالي الزبائن</p>
                                <p class="text-2xl font-bold"><?php echo count($customers); ?></p>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow p-4">
                        <div class="flex items-center">
                            <div class="p-3 rounded-full bg-green-100 text-green-600 mr-3">
                                <i class="fas fa-coins text-xl"></i>
                            </div>
                            <div>
                                <p class="text-gray-500">إجمالي الرصيد الموجب</p>
                                <p class="text-2xl font-bold text-green-600">
                                    <?php 
                                    $positive = array_sum(array_map(function($c) { 
                                        return $c['credit_balance'] > 0 ? $c['credit_balance'] : 0; 
                                    }, $customers));
                                    echo number_format($positive, 2); ?> د.م
                                </p>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow p-4">
                        <div class="flex items-center">
                            <div class="p-3 rounded-full bg-red-100 text-red-600 mr-3">
                                <i class="fas fa-file-invoice-dollar text-xl"></i>
                            </div>
                            <div>
                                <p class="text-gray-500">إجمالي الديون</p>
                                <p class="text-2xl font-bold text-red-600">
                                    <?php 
                                    $negative = array_sum(array_map(function($c) { 
                                        return $c['credit_balance'] < 0 ? abs($c['credit_balance']) : 0; 
                                    }, $customers));
                                    echo number_format($negative, 2); ?> د.م
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- قائمة الزبائن -->
                <div class="bg-white rounded-lg shadow overflow-hidden">
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">#</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">الاسم</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">عدد المعاملات</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">الرصيد</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">آخر معاملة</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">الإجراءات</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200" id="customersTableBody">
                                <?php foreach ($customers as $customer): ?>
                                <tr class="customer-card transition duration-150 ease-in-out">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo $customer['id']; ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="flex-shrink-0 h-10 w-10 bg-blue-100 rounded-full flex items-center justify-center">
                                                <i class="fas fa-user text-blue-600"></i>
                                            </div>
                                            <div class="mr-3">
                                                <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($customer['name']); ?></p>
                                                <p class="text-sm text-gray-500"><?php echo $customer['phone'] ?? 'لا يوجد'; ?></p>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 text-center">
                                        <?php echo $customer['orders_count'] ?? 0; ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium <?php echo $customer['credit_balance'] < 0 ? 'text-red-600' : 'text-green-600'; ?>">
                                        <?php echo number_format($customer['credit_balance'], 2); ?> د.م
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php 
                                        $lastOrder = getLastOrderDate($customer['id']);
                                        echo $lastOrder ? date('Y/m/d', strtotime($lastOrder)) : 'لا يوجد';
                                        ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <button onclick="showCustomerDetails(<?php echo $customer['id']; ?>)" class="text-blue-600 hover:text-blue-900 mr-3">
                                            <i class="fas fa-eye"></i> عرض
                                        </button>
                                        <button onclick="editCustomer(<?php echo $customer['id']; ?>)" class="text-yellow-600 hover:text-yellow-900 mr-3">
                                            <i class="fas fa-edit"></i> تعديل
                                        </button>
                                        <button onclick="confirmDeleteCustomer(<?php echo $customer['id']; ?>)" class="text-red-600 hover:text-red-900">
                                            <i class="fas fa-trash-alt"></i> حذف
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- نافذة إضافة/تعديل زبون -->
    <div id="customerModal" class="fixed inset-0 bg-black bg-opacity-50 hidden flex items-center justify-center z-50">
        <div class="bg-white rounded-lg w-full max-w-md mx-4">
            <div class="p-6">
                <div class="flex justify-between items-center border-b pb-4 mb-4">
                    <h3 class="text-xl font-bold text-gray-800" id="modalTitle">إضافة زبون جديد</h3>
                    <button onclick="closeModal()" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-times text-xl"></i>
                    </button>
                </div>
                <form id="customerForm">
                    <input type="hidden" id="customerId">
                    <div class="mb-4">
                        <label for="name" class="block text-sm font-medium text-gray-700 mb-1">اسم الزبون *</label>
                        <input type="text" id="name" required class="w-full p-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div class="mb-4">
                        <label for="phone" class="block text-sm font-medium text-gray-700 mb-1">رقم الهاتف</label>
                        <input type="tel" id="phone" class="w-full p-2 border rounded-lg">
                    </div>
                    <div class="mb-4">
                        <label for="address" class="block text-sm font-medium text-gray-700 mb-1">العنوان</label>
                        <textarea id="address" rows="2" class="w-full p-2 border rounded-lg"></textarea>
                    </div>
                    <div class="mb-4">
                        <label for="credit_balance" class="block text-sm font-medium text-gray-700 mb-1">الرصيد الحالي (د.م)</label>
                        <input type="number" step="0.01" id="credit_balance" class="w-full p-2 border rounded-lg">
                    </div>
                    <div class="flex justify-end space-x-3 pt-4 border-t">
                        <button type="button" onclick="closeModal()" class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-100">
                            إلغاء
                        </button>
                        <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                            <i class="fas fa-save mr-2"></i> حفظ
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- نافذة تأكيد الحذف -->
    <div id="confirmDeleteModal" class="fixed inset-0 bg-black bg-opacity-50 hidden flex items-center justify-center z-50">
        <div class="bg-white rounded-lg w-full max-w-md mx-4">
            <div class="p-6">
                <div class="flex justify-between items-center border-b pb-4 mb-4">
                    <h3 class="text-xl font-bold text-gray-800">تأكيد الحذف</h3>
                    <button onclick="closeDeleteModal()" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-times text-xl"></i>
                    </button>
                </div>
                <div class="mb-6">
                    <p class="text-gray-700">هل أنت متأكد من حذف هذا الزبون؟ سيتم حذف جميع البيانات المرتبطة به ولا يمكن استرجاعها.</p>
                </div>
                <div class="flex justify-end space-x-3 pt-4 border-t">
                    <button onclick="closeDeleteModal()" class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-100">
                        إلغاء
                    </button>
                    <button onclick="deleteCustomer()" class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                        <i class="fas fa-trash-alt mr-2"></i> حذف
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
    let currentCustomerId = null;

    // عرض نموذج إضافة زبون
    $('#addCustomerBtn').click(function() {
        $('#modalTitle').text('إضافة زبون جديد');
        $('#customerForm')[0].reset();
        $('#customerId').val('');
        $('#customerModal').removeClass('hidden');
    });

    // عرض نموذج تعديل زبون
    function editCustomer(id) {
        $.ajax({
            url: 'api/customers.php?id=' + id,
            method: 'GET',
            success: function(response) {
                if (response.success) {
                    $('#modalTitle').text('تعديل زبون');
                    $('#customerId').val(response.data.id);
                    $('#name').val(response.data.name);
                    $('#phone').val(response.data.phone || '');
                    $('#address').val(response.data.address || '');
                    $('#credit_balance').val(response.data.credit_balance || 0);
                    $('#customerModal').removeClass('hidden');
                } else {
                    alert(response.message || 'فشل في جلب بيانات الزبون');
                }
            },
            error: function() {
                alert('حدث خطأ في الاتصال بالخادم');
            }
        });
    }

    // عرض تفاصيل الزبون
    function showCustomerDetails(id) {
        window.location.href = 'customer_details.php?id=' + id;
    }

    // تأكيد حذف الزبون
    function confirmDeleteCustomer(id) {
        currentCustomerId = id;
        $('#confirmDeleteModal').removeClass('hidden');
    }

    // حذف الزبون
    function deleteCustomer() {
        if (!currentCustomerId) return;
        
        $.ajax({
            url: 'api/customers.php',
            method: 'DELETE',
            data: { id: currentCustomerId },
            beforeSend: function() {
                $('#confirmDeleteModal button').prop('disabled', true);
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.message || 'فشل في حذف الزبون');
                }
            },
            error: function() {
                alert('حدث خطأ في الاتصال بالخادم');
            },
            complete: function() {
                $('#confirmDeleteModal button').prop('disabled', false);
                closeDeleteModal();
            }
        });
    }

    // إغلاق النوافذ
    function closeModal() {
        $('#customerModal').addClass('hidden');
    }

    function closeDeleteModal() {
        $('#confirmDeleteModal').addClass('hidden');
        currentCustomerId = null;
    }

    // إرسال نموذج الزبون
    $('#customerForm').submit(function(e) {
        e.preventDefault();
        
        const formData = {
            id: $('#customerId').val(),
            name: $('#name').val(),
            phone: $('#phone').val(),
            address: $('#address').val(),
            credit_balance: $('#credit_balance').val()
        };

        const method = formData.id ? 'PUT' : 'POST';
        const url = 'api/customers.php';

        $.ajax({
            url: url,
            method: method,
            contentType: 'application/json',
            data: JSON.stringify(formData),
            beforeSend: function() {
                $('#customerForm button[type="submit"]').prop('disabled', true)
                    .html('<i class="fas fa-spinner fa-spin mr-2"></i> جاري الحفظ');
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.message || 'فشل في حفظ البيانات');
                }
            },
            error: function() {
                alert('حدث خطأ في الاتصال بالخادم');
            },
            complete: function() {
                $('#customerForm button[type="submit"]').prop('disabled', false)
                    .html('<i class="fas fa-save mr-2"></i> حفظ');
            }
        });
    });

    // فلترة الزبائن
    $('#searchByName, #filterByCredit').on('input change', function() {
        const searchTerm = $('#searchByName').val().toLowerCase();
        const creditFilter = $('#filterByCredit').val();
        
        $('.customer-card').each(function() {
            const name = $(this).find('td:nth-child(2)').text().toLowerCase();
            const creditText = $(this).find('td:nth-child(4)').text();
            const credit = parseFloat(creditText.replace(/[^0-9.-]/g, ''));
            
            let nameMatch = name.includes(searchTerm);
            let creditMatch = true;
            
            if (creditFilter === 'positive') creditMatch = credit > 0;
            else if (creditFilter === 'negative') creditMatch = credit < 0;
            else if (creditFilter === 'zero') creditMatch = credit === 0;
            
            $(this).toggle(nameMatch && creditMatch);
        });
    });

    // إعادة تعيين الفلتر
    $('#resetFilters').click(function() {
        $('#searchByName').val('');
        $('#filterByCredit').val('all');
        $('.customer-card').show();
    });
    </script>
</body>
</html>