<?php
header('Content-Type: application/json');
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../functions.php';

// Start the session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$pdo = getDBConnection();
$response = ['success' => false, 'message' => ''];

try {
    if (!isset($_SESSION['user_id'])) {
        $response['message'] = 'غير مسموح بالوصول';
        http_response_code(401);
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
        exit();
    }

    $method = $_SERVER['REQUEST_METHOD'];

    switch ($method) {
        case 'GET':
            if (isset($_GET['id'])) {
                $customer = getCustomer($_GET['id']);
                if ($customer) {
                    $response = [
                        'success' => true,
                        'data' => $customer
                    ];
                } else {
                    $response['message'] = 'الزبون غير موجود';
                    http_response_code(404);
                }
            } else {
                $response['message'] = 'معرّف الزبون مطلوب';
                http_response_code(400);
            }
            break;

        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true) ?: $_POST;
            
            $columns = [];
            $values = [];
            $placeholders = [];
            
            if (empty($data['name'])) {
                $response['message'] = 'اسم الزبون مطلوب';
                http_response_code(400);
                break;
            }
            
            $columns[] = 'name';
            $values[] = trim($data['name']);
            $placeholders[] = '?';
            
            $optionalFields = ['credit_balance', 'discounts'];
            foreach ($optionalFields as $field) {
                if (isset($data[$field])) {
                    $columns[] = $field;
                    $values[] = ($field === 'discounts') ? json_encode($data[$field], JSON_UNESCAPED_UNICODE) : $data[$field];
                    $placeholders[] = '?';
                }
            }
            
            $sql = "INSERT INTO customers (" . implode(', ', $columns) . ") 
                    VALUES (" . implode(', ', $placeholders) . ")";
            $stmt = $pdo->prepare($sql);
            $success = $stmt->execute($values);
            
            if ($success) {
                $response = [
                    'success' => true,
                    'message' => 'تم إضافة الزبون بنجاح',
                    'data' => ['id' => $pdo->lastInsertId()]
                ];
            } else {
                $response['message'] = 'فشل في إضافة الزبون';
                http_response_code(500);
            }
            break;

        case 'PUT':
            parse_str(file_get_contents("php://input"), $data);
            $data = $data ?: [];

            if (empty($data['id']) || empty($data['name'])) {
                $response['message'] = 'معرّف واسم الزبون مطلوبين';
                http_response_code(400);
                break;
            }

            $updates = [];
            $values = [];
            $updates[] = 'name = ?';
            $values[] = trim($data['name']);
            
            if (isset($data['credit_balance'])) { $updates[] = 'credit_balance = ?'; $values[] = $data['credit_balance']; }
            if (isset($data['discounts'])) { $updates[] = 'discounts = ?'; $values[] = json_encode($data['discounts'], JSON_UNESCAPED_UNICODE); }
            
            $values[] = $data['id'];

            $sql = "UPDATE customers SET " . implode(', ', $updates) . " WHERE id = ?";
            $stmt = $pdo->prepare($sql);
            $success = $stmt->execute($values);
            
            if ($success) {
                $response = [
                    'success' => true,
                    'message' => 'تم تحديث الزبون بنجاح'
                ];
            } else {
                $response['message'] = 'فشل في تحديث الزبون';
                http_response_code(500);
            }
            break;

        case 'DELETE':
            parse_str(file_get_contents("php://input"), $data);
            $id = $data['id'] ?? null;

            if ($id) {
                $stmt = $pdo->prepare("DELETE FROM customers WHERE id = ?");
                $success = $stmt->execute([$id]);

                if ($success) {
                    $response = [
                        'success' => true,
                        'message' => 'تم حذف الزبون بنجاح'
                    ];
                } else {
                    $response['message'] = 'فشل في حذف الزبون';
                    http_response_code(500);
                }
            } else {
                $response['message'] = 'معرّف الزبون مطلوب';
                http_response_code(400);
            }
            break;

        default:
            $response['message'] = 'طريقة طلب غير صالحة';
            http_response_code(405);
            break;
    }
} catch (PDOException $e) {
    $response['message'] = 'خطأ في قاعدة البيانات: ' . $e->getMessage();
    http_response_code(500);
    error_log("PDO Error: " . $e->getMessage());
}

echo json_encode($response, JSON_UNESCAPED_UNICODE);
?>