<?php
session_start();
header('Content-Type: application/json');
require_once __DIR__ . '/../config.php'; // تأكد من المسار الصحيح لـ config.php
require_once __DIR__ . '/../functions.php'; // تأكد من المسار الصحيح لـ functions.php

$pdo = getDBConnection();
$response = ['success' => false, 'message' => ''];

try {
    // التحقق من تسجيل الدخول
    if (!isset($_SESSION['user_id'])) {
        $response['message'] = 'غير مسموح بالوصول';
        echo json_encode($response);
        exit();
    }

    // التحقق من أن الطلب هو POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        $response['message'] = 'Invalid request method. Only POST is allowed.';
        echo json_encode($response);
        exit();
    }

    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    // التحقق من البيانات الأساسية للمعاملة
    if (empty($data['items']) || !isset($data['total']) || !isset($data['payment_method'])) {
        $response['message'] = 'بيانات المعاملة غير مكتملة.';
        echo json_encode($response);
        exit();
    }

    $pdo->beginTransaction(); // بدء المعاملة

    $customer_id = $data['customer_id'] ?? null;
    $customer_name = $data['customer_name'] ?? 'زبون عابر';
    $total = (float)$data['total'];
    $payment_method = $data['payment_method'];
    $status = $payment_method === 'cash' ? 'paid' : 'unpaid';
    $notes = $data['notes'] ?? null;

    // حساب إجمالي التكلفة (total_cost) بناءً على المنتجات وكمياتها
    $total_cost = 0;
    if (!empty($data['items'])) {
        $product_ids = array_column($data['items'], 'id');
        if (!empty($product_ids)) {
            $placeholders = implode(',', array_fill(0, count($product_ids), '?'));
            $stmt_product_costs = $pdo->prepare("SELECT id, cost_price FROM products WHERE id IN ($placeholders)");
            $stmt_product_costs->execute($product_ids);
            $product_costs = $stmt_product_costs->fetchAll(PDO::FETCH_KEY_PAIR); // Fetch as [id => cost_price]

            foreach ($data['items'] as $item) {
                $cost_price = $product_costs[$item['id']] ?? 0;
                $total_cost += $cost_price * $item['quantity'];
            }
        }
    }


    // إدخال المعاملة في جدول transactions
    $stmt = $pdo->prepare("
        INSERT INTO transactions 
        (customer_id, customer_name, total, total_cost, payment_method, status, notes, date) 
        VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    $stmt->execute([
        $customer_id,
        $customer_name,
        $total,
        $total_cost,
        $payment_method,
        $status,
        $notes
    ]);

    $transaction_id = $pdo->lastInsertId(); // الحصول على ID المعاملة الجديدة

    // إدخال تفاصيل المعاملة في جدول transaction_items
    $stmt_item = $pdo->prepare("
        INSERT INTO transaction_items 
        (transaction_id, product_id, name, quantity, price, discount_amount) 
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    foreach ($data['items'] as $item) {
        $stmt_item->execute([
            $transaction_id,
            $item['id'],
            $item['name'],
            $item['quantity'],
            $item['price'],
            $item['discount']
        ]);
    }

    $remaining_credit = 0;
    // تحديث رصيد الزبون إذا كانت طريقة الدفع دين
    if ($payment_method === 'credit' && $customer_id) {
        $stmt_customer = $pdo->prepare("SELECT credit_balance FROM customers WHERE id = ?");
        $stmt_customer->execute([$customer_id]);
        $customer = $stmt_customer->fetch(PDO::FETCH_ASSOC);
        $current_credit = $customer['credit_balance'] ?? 0;
        
        $new_credit = $current_credit + $total;

        $stmt_update_credit = $pdo->prepare("
            UPDATE customers SET credit_balance = ? WHERE id = ?
        ");
        $stmt_update_credit->execute([$new_credit, $customer_id]);
        $remaining_credit = $new_credit;
    }

    $pdo->commit(); // تأكيد المعاملة

    $response = [
        'success' => true,
        'message' => 'Transaction completed successfully.',
        'transaction_id' => $transaction_id, // إرجاع transaction_id
        'remaining_credit' => $remaining_credit
    ];

} catch (PDOException $e) {
    $pdo->rollBack(); // التراجع عن المعاملة عند الخطأ
    $response['message'] = 'فشل في إتمام العملية: ' . $e->getMessage();
    error_log("Error processing transaction: " . $e->getMessage()); // تسجيل الخطأ
} catch (Exception $e) {
    $pdo->rollBack();
    $response['message'] = 'حدث خطأ غير متوقع: ' . $e->getMessage();
    error_log("Unexpected error: " . $e->getMessage());
}

echo json_encode($response); // إرجاع الاستجابة JSON
?>