<?php
session_start();
require 'db.php';

// Vérification de la connexion
if (!isset($_SESSION['user']['id']) || empty($_SESSION['user']['id'])) {
    $_SESSION['login_error'] = "Vous devez vous connecter d'abord";
    header('Location: login.php');
    exit();
}

// Traitement de la modification du statut
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['colis_id'], $_POST['nouveau_statut'])) {
    $colis_id = (int)$_POST['colis_id'];
    $nouveau_statut = trim($_POST['nouveau_statut']);
    
    // Liste des statuts autorisés
    $statuts_autorises = ['en attente', 'en cours', 'livré', 'annulé', 'prêt pour livraison', 'ramassé', 'expédié', 'retourné'];
    
    if (in_array($nouveau_statut, $statuts_autorises)) {
        $stmt = $pdo->prepare("UPDATE colis SET statut = ? WHERE id = ?");
        $stmt->execute([$nouveau_statut, $colis_id]);
        $_SESSION['success_message'] = "Statut mis à jour avec succès";
        header("Location: ".$_SERVER['PHP_SELF']);
        exit();
    }
}

// Récupération des villes du livreur
$livreur_id = (int)$_SESSION['user']['id'];
$sql_villes = "SELECT ville FROM user_villes WHERE user_id = ?";
$stmt_villes = $pdo->prepare($sql_villes);
$stmt_villes->execute([$livreur_id]);
$villes_livreur = $stmt_villes->fetchAll(PDO::FETCH_COLUMN);

// Filtres de recherche
$recherche = isset($_GET['recherche']) ? trim($_GET['recherche']) : '';
$filtre = isset($_GET['filtre']) ? trim($_GET['filtre']) : '';

// Construction de la requête (on ne cache plus les colis livrés)
$sql = "SELECT * FROM colis WHERE 1=1";
$params = [];

if (!empty($villes_livreur)) {
    $placeholders = implode(',', array_fill(0, count($villes_livreur), '?'));
    $sql .= " AND ville IN ($placeholders)";
    $params = array_merge($params, $villes_livreur);
}

if (!empty($recherche)) {
    $sql .= " AND (nom LIKE ? OR prenom LIKE ? OR telephone LIKE ? OR num_colis LIKE ?)";
    $search_term = "%$recherche%";
    array_push($params, $search_term, $search_term, $search_term, $search_term);
}

if (!empty($filtre)) {
    $sql .= " AND statut = ?";
    $params[] = $filtre;
}

$sql .= " ORDER BY id DESC";

// Exécution de la requête
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$colis = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tableau de Bord Livreur | Colis</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --secondary: #3a0ca3;
            --light: #f8f9fa;
            --dark: #212529;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Poppins', sans-serif;
        }

        body {
            background-color: #f5f7fa;
        }

        .dashboard {
            display: flex;
            min-height: 100vh;
        }

        /* Sidebar */
        .sidebar {
            width: 280px;
            background: white;
            box-shadow: 2px 0 15px rgba(0, 0, 0, 0.1);
        }

        .sidebar-header {
            padding: 20px;
            border-bottom: 1px solid #eee;
        }

        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 10px;
            color: var(--primary);
            font-size: 1.2rem;
            font-weight: 700;
        }

        .sidebar-menu {
            list-style: none;
            padding: 15px 0;
        }

        .sidebar-menu li a {
            display: flex;
            align-items: center;
            padding: 12px 20px;
            color: #555;
            text-decoration: none;
            transition: all 0.3s;
            gap: 10px;
        }

        .sidebar-menu li a:hover,
        .sidebar-menu li a.active {
            background: rgba(67, 97, 238, 0.1);
            color: var(--primary);
            border-left: 3px solid var(--primary);
        }

        /* Main Content */
        .main-content {
            flex: 1;
            padding: 30px;
        }

        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
        }

        .page-title {
            color: var(--dark);
            font-size: 1.5rem;
            font-weight: 700;
        }

        /* Search Filter */
        .search-filter {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }

        .search-form {
            display: flex;
            gap: 10px;
        }

        .search-input {
            flex: 1;
            padding: 10px 15px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 1rem;
        }

        .filter-select {
            padding: 10px 15px;
            border: 1px solid #ddd;
            border-radius: 6px;
            min-width: 180px;
        }

        .search-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 0 20px;
            border-radius: 6px;
            cursor: pointer;
            transition: all 0.3s;
        }

        .search-btn:hover {
            background: var(--secondary);
        }

        /* Colis Table */
        .colis-table {
            width: 100%;
            background: white;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            overflow: hidden;
            border-collapse: collapse;
        }

        .colis-table th {
            background: var(--primary);
            color: white;
            padding: 15px;
            text-align: left;
            font-weight: 500;
        }

        .colis-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #eee;
        }

        .colis-table tr:last-child td {
            border-bottom: none;
        }

        .colis-table tr:hover td {
            background: rgba(67, 97, 238, 0.05);
        }

        /* Status Badges */
        .badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 500;
            display: inline-block;
        }

        .badge-warning {
            background: rgba(255, 193, 7, 0.2);
            color: #d39e00;
        }

        .badge-primary {
            background: rgba(67, 97, 238, 0.2);
            color: var(--primary);
        }

        .badge-success {
            background: rgba(40, 167, 69, 0.2);
            color: var(--success);
        }

        .badge-danger {
            background: rgba(220, 53, 69, 0.2);
            color: var(--danger);
        }

        .badge-info {
            background: rgba(23, 162, 184, 0.2);
            color: var(--info);
        }

        /* Buttons */
        .btn {
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 0.8rem;
            text-decoration: none;
            transition: all 0.3s;
            cursor: pointer;
            border: none;
        }

        .btn-sm {
            padding: 3px 8px;
            font-size: 0.75rem;
        }

        .btn-primary {
            background: rgba(67, 97, 238, 0.2);
            color: var(--primary);
        }

        .btn-primary:hover {
            background: rgba(67, 97, 238, 0.3);
        }

        /* Modal */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
        }

        .modal-content {
            background-color: white;
            margin: 10% auto;
            padding: 20px;
            border-radius: 8px;
            width: 400px;
            max-width: 90%;
            box-shadow: 0 5px 15px rgba(0,0,0,0.3);
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .modal-title {
            font-size: 1.2rem;
            font-weight: 600;
        }

        .close-btn {
            font-size: 1.5rem;
            cursor: pointer;
            color: #aaa;
        }

        .close-btn:hover {
            color: var(--dark);
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
        }

        .form-select {
            width: 100%;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }

        .modal-footer {
            display: flex;
            justify-content: flex-end;
            gap: 10px;
            margin-top: 20px;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-success {
            background: var(--success);
            color: white;
        }

        /* Alerts */
        .alert {
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-info {
            background: rgba(23, 162, 184, 0.2);
            color: #0c5460;
        }

        .alert-warning {
            background: rgba(255, 193, 7, 0.2);
            color: #856404;
        }

        .alert-success {
            background: rgba(40, 167, 69, 0.2);
            color: #155724;
        }

        /* Responsive */
        @media (max-width: 992px) {
            .dashboard {
                flex-direction: column;
            }
            .sidebar {
                width: 100%;
            }
            .search-form {
                flex-direction: column;
            }
            .filter-select {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="dashboard">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">
                    <i class="fas fa-shipping-fast"></i>
                    <span>Espace Livreur</span>
                </div>
            </div>
            <ul class="sidebar-menu">
                <li><a href="index.php" class="active"><i class="fas fa-box-open"></i> Colis</a></li>
                
                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a></li>
            </ul>
        </div>

        <!-- Main Content -->
        <div class="main-content">
            <div class="page-header">
                <h1 class="page-title">
                    <i class="fas fa-boxes"></i> Colis Disponibles
                </h1>
                <div class="badge badge-primary">
                    Villes: <?= implode(', ', $villes_livreur) ?>
                </div>
            </div>

            <!-- Search Filter -->
            <div class="search-filter">
                <form method="get" class="search-form">
                    <input type="text" name="recherche" class="search-input" 
                           placeholder="Rechercher par N° Colis, nom, téléphone..." value="<?= htmlspecialchars($recherche) ?>">
                    <select name="filtre" class="filter-select">
                        <option value="">Tous les statuts</option>
                        <option value="en attente" <?= $filtre === 'en attente' ? 'selected' : '' ?>>En attente</option>
                        <option value="en cours" <?= $filtre === 'en cours' ? 'selected' : '' ?>>En cours</option>
                        <option value="prêt pour livraison" <?= $filtre === 'prêt pour livraison' ? 'selected' : '' ?>>Prêt pour livraison</option>
                        <option value="ramassé" <?= $filtre === 'ramassé' ? 'selected' : '' ?>>Ramassé</option>
                        <option value="expédié" <?= $filtre === 'expédié' ? 'selected' : '' ?>>Expédié</option>
                        <option value="livré" <?= $filtre === 'livré' ? 'selected' : '' ?>>Livré</option>
                        <option value="retourné" <?= $filtre === 'retourné' ? 'selected' : '' ?>>Retourné</option>
                        <option value="annulé" <?= $filtre === 'annulé' ? 'selected' : '' ?>>Annulé</option>
                    </select>
                    <button type="submit" class="search-btn">
                        <i class="fas fa-search"></i> Filtrer
                    </button>
                </form>
            </div>

            <!-- Messages -->
            <?php if (isset($_SESSION['success_message'])): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?= $_SESSION['success_message'] ?>
                </div>
                <?php unset($_SESSION['success_message']); ?>
            <?php endif; ?>

            <?php if (empty($villes_livreur)): ?>
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> Aucune ville n'est enregistrée pour votre compte. Veuillez contacter l'administrateur.
                </div>
            <?php elseif (!empty($colis)): ?>
                <div class="table-responsive">
                    <table class="colis-table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>N° Colis</th>
                                <th>Nom</th>
                                <th>Prénom</th>
                                <th>Ville</th>
                                <th>Téléphone</th>
                                <th>Montant</th>
                                <th>Statut</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($colis as $item): ?>
                            <tr>
                                <td><?= $item['id'] ?></td>
                                <td><?= htmlspecialchars($item['num_colis']) ?></td>
                                <td><?= htmlspecialchars($item['nom']) ?></td>
                                <td><?= htmlspecialchars($item['prenom']) ?></td>
                                <td><?= htmlspecialchars($item['ville']) ?></td>
                                <td><?= htmlspecialchars($item['telephone']) ?></td>
                                <td><?= htmlspecialchars($item['montant']) ?> DH</td>
                                <td>
                                    <?php
                                    $badge_class = '';
                                    switch ($item['statut']) {
                                        case 'en attente': $badge_class = 'badge-warning'; break;
                                        case 'en cours': $badge_class = 'badge-primary'; break;
                                        case 'prêt pour livraison': $badge_class = 'badge-info'; break;
                                        case 'ramassé': $badge_class = 'badge-primary'; break;
                                        case 'expédié': $badge_class = 'badge-info'; break;
                                        case 'livré': $badge_class = 'badge-success'; break;
                                        case 'retourné': $badge_class = 'badge-warning'; break;
                                        case 'annulé': $badge_class = 'badge-danger'; break;
                                        default: $badge_class = 'badge-primary';
                                    }
                                    ?>
                                    <span class="badge <?= $badge_class ?>">
                                        <?= htmlspecialchars($item['statut']) ?>
                                    </span>
                                </td>
                                <td>
                                    <button onclick="openModal(<?= $item['id'] ?>, '<?= htmlspecialchars($item['statut']) ?>')" 
                                            class="btn btn-primary btn-sm">
                                        <i class="fas fa-edit"></i> Modifier
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div class="alert alert-warning">
                    <i class="fas fa-box-open"></i> Aucun colis disponible dans vos villes actuellement.
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Modal de modification du statut -->
    <div id="statusModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Modifier le statut du colis</h3>
                <span class="close-btn" onclick="closeModal()">&times;</span>
            </div>
            <form method="post" id="statusForm">
                <input type="hidden" name="colis_id" id="colis_id">
                <div class="form-group">
                    <label for="nouveau_statut" class="form-label">Nouveau statut:</label>
                    <select name="nouveau_statut" id="nouveau_statut" class="form-select" required>
                        <option value="en attente">En attente</option>
                        <option value="en cours">En cours</option>
                        <option value="prêt pour livraison">Prêt pour livraison</option>
                        <option value="ramassé">Ramassé</option>
                        <option value="expédié">Expédié</option>
                        <option value="livré">Livré</option>
                        <option value="retourné">Retourné</option>
                        <option value="annulé">Annulé</option>
                    </select>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annuler</button>
                    <button type="submit" class="btn btn-success">Enregistrer</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Gestion de la modal
        function openModal(colisId, currentStatus) {
            document.getElementById('colis_id').value = colisId;
            document.getElementById('nouveau_statut').value = currentStatus;
            document.getElementById('statusModal').style.display = 'block';
        }

        function closeModal() {
            document.getElementById('statusModal').style.display = 'none';
        }

        // Fermer la modal en cliquant à l'extérieur
        window.onclick = function(event) {
            const modal = document.getElementById('statusModal');
            if (event.target === modal) {
                closeModal();
            }
        }
    </script>
</body>
</html>