<?php
// تمكين عرض الأخطاء للتطوير
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// التحقق من تسجيل الدخول
if (!isset($_SESSION['user']['id'])) {
    $_SESSION['login_error'] = "يجب تسجيل الدخول أولاً";
    header('Location: login.php');
    exit();
}

if ($_SESSION['user']['role'] !== 'admin') {
    die("Accès refusé: Vous n'avez pas les permissions nécessaires.");
}

require 'db.php';

// إنشاء جدول الأسعار إذا لم يكن موجوداً
try {
    $tableExists = $pdo->query("SHOW TABLES LIKE 'client_prices'")->rowCount() > 0;
    
    if (!$tableExists) {
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS client_prices (
                id INT AUTO_INCREMENT PRIMARY KEY,
                client_id INT NOT NULL,
                ville VARCHAR(100) NOT NULL,
                prix DECIMAL(10,2) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (client_id) REFERENCES users(id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");
    }
} catch (PDOException $e) {
    die("Erreur lors de la création de la table: " . $e->getMessage());
}

// جلب بيانات الزبائن والمدن
try {
    // جلب بيانات الزبائن (المستخدمين بدور "client")
    $clients_stmt = $pdo->query("SELECT id, nom FROM users WHERE role = 'client'");
    if ($clients_stmt === false) {
        throw new PDOException("Erreur lors de la récupération des clients");
    }
    $clients = $clients_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // جلب بيانات المدن
    $villes_stmt = $pdo->query("SELECT nom_ville FROM villes");
    if ($villes_stmt === false) {
        throw new PDOException("Erreur lors de la récupération des villes");
    }
    $villes = $villes_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // معالجة إضافة سعر جديد
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_price'])) {
        $client_id = (int)$_POST['client'];
        $ville = htmlspecialchars($_POST['ville']);
        $prix = (float)$_POST['prix'];
        
        // التحقق من صحة البيانات
        if ($client_id <= 0 || empty($ville) || $prix <= 0) {
            throw new PDOException("Données invalides");
        }
        
        $stmt = $pdo->prepare("INSERT INTO client_prices (client_id, ville, prix) VALUES (?, ?, ?)");
        if ($stmt === false) {
            throw new PDOException("Erreur de préparation de la requête");
        }
        
        $result = $stmt->execute([$client_id, $ville, $prix]);
        if (!$result) {
            throw new PDOException("Erreur lors de l'ajout du prix");
        }
        
        $_SESSION['success_message'] = "Prix client ajouté avec succès";
        header('Location: remises.php');
        exit();
    }
    
    // جلب بيانات الأسعار الحالية
    $prices = [];
    $prices_stmt = $pdo->query("
        SELECT cp.*, u.nom 
        FROM client_prices cp
        JOIN users u ON cp.client_id = u.id
        ORDER BY cp.id DESC
    ");
    if ($prices_stmt !== false) {
        $prices = $prices_stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
} catch (PDOException $e) {
    die("Erreur de base de données: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion Prix Clients - My Delivery</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #4361ee;
            --secondary-color: #3a0ca3;
            --accent-color: #4cc9f0;
            --light-color: #f8f9fa;
            --dark-color: #212529;
            --success-color: #4bb543;
            --error-color: #ff3333;
            --warning-color: #f8d210;
            --admin-color: #6a1b9a;
            --border-radius: 12px;
            --box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: #f5f7fa;
            color: var(--dark-color);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        .admin-sidebar {
            width: 280px;
            background: var(--admin-color);
            color: white;
            padding: 20px 0;
        }
        
        .admin-sidebar-header {
            padding: 0 20px 20px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            margin-bottom: 20px;
        }
        
        .admin-logo {
            font-size: 1.5rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .admin-logo i {
            font-size: 2rem;
        }
        
        .admin-menu {
            list-style: none;
        }
        
        .admin-menu li a {
            display: flex;
            align-items: center;
            padding: 15px 25px;
            color: rgba(255,255,255,0.8);
            text-decoration: none;
            transition: all 0.3s ease;
            gap: 12px;
            font-size: 0.95rem;
        }
        
        .admin-menu li a:hover, 
        .admin-menu li a.active {
            background: rgba(255,255,255,0.1);
            color: white;
            border-left: 3px solid white;
        }
        
        .admin-menu li a i {
            width: 20px;
            text-align: center;
            font-size: 1.1rem;
        }
        
        .admin-main {
            flex: 1;
            padding: 30px;
            background: #f5f7fa;
        }
        
        .admin-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid #eee;
        }
        
        .admin-title {
            font-size: 1.8rem;
            font-weight: 600;
            color: var(--dark-color);
        }
        
        .admin-user {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .admin-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: var(--primary-color);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
        }
        
        .admin-section {
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--box-shadow);
            padding: 25px;
            margin-bottom: 30px;
        }
        
        .section-title {
            font-size: 1.2rem;
            font-weight: 600;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            color: var(--admin-color);
        }
        
        .section-title i {
            font-size: 1.4rem;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: #555;
        }
        
        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid #ddd;
            border-radius: var(--border-radius);
            font-family: 'Poppins', sans-serif;
            font-size: 1rem;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: var(--admin-color);
            outline: none;
            box-shadow: 0 0 0 3px rgba(106, 27, 154, 0.1);
        }
        
        .btn {
            display: inline-block;
            padding: 12px 25px;
            background: var(--admin-color);
            color: white;
            border: none;
            border-radius: var(--border-radius);
            font-family: 'Poppins', sans-serif;
            font-size: 1rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn:hover {
            background: #5a189a;
            transform: translateY(-2px);
        }
        
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background-color: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            border: 1px solid #f5c6cb;
        }
        
        .prices-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        
        .prices-table th {
            text-align: left;
            padding: 12px 15px;
            background: #f8f9fa;
            font-weight: 500;
            color: #555;
        }
        
        .prices-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #eee;
        }
        
        .prices-table tr:last-child td {
            border-bottom: none;
        }
        
        .action-btn {
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 0.8rem;
            text-decoration: none;
            margin-right: 5px;
            border: none;
            cursor: pointer;
            font-family: 'Poppins', sans-serif;
        }
        
        .edit-btn {
            background: rgba(248, 210, 16, 0.2);
            color: #b38f00;
        }
        
        .edit-btn:hover {
            background: rgba(248, 210, 16, 0.3);
        }
        
        .delete-btn {
            background: rgba(255, 51, 51, 0.2);
            color: var(--error-color);
        }
        
        .delete-btn:hover {
            background: rgba(255, 51, 51, 0.3);
        }
        
        .no-data {
            text-align: center;
            padding: 20px;
            color: #777;
        }
        
        .action-form {
            display: inline;
        }
        
        @media (max-width: 992px) {
            .admin-container {
                flex-direction: column;
            }
            
            .admin-sidebar {
                width: 100%;
                height: auto;
            }
        }
        
        @media (max-width: 768px) {
            .admin-main {
                padding: 15px;
            }
            
            .prices-table {
                display: block;
                overflow-x: auto;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- القائمة الجانبية -->
        <div class="admin-sidebar">
            <div class="admin-sidebar-header">
                <div class="admin-logo">
                    <i class="fas fa-user-shield"></i>
                    <span>Admin Panel</span>
                </div>
            </div>
            <ul class="admin-menu">
                <li><a href="index.php"><i class="fas fa-tachometer-alt"></i> Tableau de Bord</a></li>
                <li><a href="admin_users.php"><i class="fas fa-users"></i> Gestion Utilisateurs</a></li>
                <li><a href="admin_colis.php"><i class="fas fa-boxes"></i> Gestion Colis</a></li>
                <li><a href="admin_villes.php"><i class="fas fa-city"></i> Gestion Villes & Tarifs</a></li>
                <li><a href="admin_commissions.php"><i class="fas fa-money-bill-wave"></i> Commissions Livreurs</a></li>
                <li><a href="remises.php" class="active"><i class="fas fa-tag"></i> Remise Clients</a></li>
               
                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a></li>
            </ul>
        </div>
        
        <!-- المحتوى الرئيسي -->
        <div class="admin-main">
            <div class="admin-header">
                <h1 class="admin-title"><i class="fas fa-tag"></i> Gestion des Prix Clients</h1>
                <div class="admin-user">
                    <div class="admin-avatar">
                        <?= substr($_SESSION['user']['nom'] ?? 'A', 0, 1) ?>
                    </div>
                    <span><?= $_SESSION['user']['nom'] ?? 'Admin' ?></span>
                </div>
            </div>
            
            <?php if (isset($_SESSION['success_message'])): ?>
                <div class="alert-success">
                    <i class="fas fa-check-circle"></i> <?= $_SESSION['success_message'] ?>
                </div>
                <?php unset($_SESSION['success_message']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['error_message'])): ?>
                <div class="alert-error">
                    <i class="fas fa-exclamation-circle"></i> <?= $_SESSION['error_message'] ?>
                </div>
                <?php unset($_SESSION['error_message']); ?>
            <?php endif; ?>
            
            <!-- نموذج إضافة سعر جديد -->
            <div class="admin-section">
                <h3 class="section-title">
                    <i class="fas fa-plus-circle"></i> Ajouter un Prix Client
                </h3>
                
                <form method="POST" action="">
                    <div class="form-group">
                        <label for="client">Client</label>
                        <select class="form-control" id="client" name="client" required>
                            <option value="">Sélectionner un client</option>
                            <?php foreach ($clients as $client): ?>
                                <option value="<?= htmlspecialchars($client['id']) ?>">
                                    <?= htmlspecialchars($client['nom']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="ville">Ville</label>
                        <select class="form-control" id="ville" name="ville" required>
                            <option value="">Sélectionner une ville</option>
                            <?php foreach ($villes as $ville): ?>
                                <option value="<?= htmlspecialchars($ville['nom_ville']) ?>">
                                    <?= htmlspecialchars($ville['nom_ville']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="prix">Prix (DH)</label>
                        <input type="number" class="form-control" id="prix" name="prix" 
                               step="0.01" min="0" required placeholder="Prix spécial pour le client">
                    </div>
                    
                    <button type="submit" name="submit_price" class="btn">
                        <i class="fas fa-save"></i> Enregistrer
                    </button>
                </form>
            </div>
            
            <!-- قائمة الأسعار الحالية -->
            <div class="admin-section">
                <h3 class="section-title">
                    <i class="fas fa-list"></i> Liste des Prix Clients
                </h3>
                
                <?php if (empty($prices)): ?>
                    <p class="no-data">Aucun prix client enregistré pour le moment.</p>
                <?php else: ?>
                    <table class="prices-table">
                        <thead>
                            <tr>
                                <th>Client</th>
                                <th>Ville</th>
                                <th>Prix (DH)</th>
                                <th>Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($prices as $price): ?>
                            <tr>
                                <td><?= htmlspecialchars($price['nom']) ?></td>
                                <td><?= htmlspecialchars($price['ville']) ?></td>
                                <td><?= number_format($price['prix'], 2) ?> DH</td>
                                <td><?= date('d/m/Y H:i', strtotime($price['created_at'])) ?></td>
                                <td>
                                    <a href="edit_client_price.php?id=<?= $price['id'] ?>" class="action-btn edit-btn">
                                        <i class="fas fa-edit"></i> Modifier
                                    </a>
                                    <form method="POST" action="delete_client_price.php" class="action-form">
                                        <input type="hidden" name="id" value="<?= $price['id'] ?>">
                                        <button type="submit" class="action-btn delete-btn" onclick="return confirm('Êtes-vous sûr de vouloir supprimer ce prix client?')">
                                            <i class="fas fa-trash"></i> Supprimer
                                        </button>
                                    </form>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
            
            <!-- حقوق التصميم -->
            <footer class="admin-footer">
                <p>&copy; <?= date('Y') ?> My Delivery. Tous droits réservés.</p>
            </footer>
        </div>
    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // تأكيد قبل الحذف
            document.querySelectorAll('.delete-btn').forEach(btn => {
                btn.addEventListener('click', function(e) {
                    if (!confirm('Êtes-vous sûr de vouloir supprimer ce prix client?')) {
                        e.preventDefault();
                    }
                });
            });
        });
    </script>
</body>
</html>