<?php
// تمكين عرض الأخطاء للتطوير
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// التحقق من تسجيل الدخول والصلاحيات
if (!isset($_SESSION['user']['id']) || $_SESSION['user']['role'] !== 'admin') {
    header('Location: login.php');
    exit();
}

require 'db.php';

// التحقق من وجود معرف السعر في الرابط
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $_SESSION['error_message'] = "ID de prix client invalide";
    header('Location: remises.php');
    exit();
}

$price_id = (int)$_GET['id'];

// جلب بيانات السعر الحالي
try {
    $stmt = $pdo->prepare("
        SELECT cp.*, u.nom 
        FROM client_prices cp
        JOIN users u ON cp.client_id = u.id
        WHERE cp.id = ?
    ");
    $stmt->execute([$price_id]);
    $price = $stmt->fetch();

    if (!$price) {
        $_SESSION['error_message'] = "Prix client non trouvé";
        header('Location: remises.php');
        exit();
    }

    // جلب بيانات الزبائن والمدن
    $clients_stmt = $pdo->query("SELECT id, nom FROM users WHERE role = 'client'");
    $clients = $clients_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $villes_stmt = $pdo->query("SELECT nom_ville FROM villes");
    $villes = $villes_stmt->fetchAll(PDO::FETCH_ASSOC);

    // معالجة تحديث السعر
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_price'])) {
        $client_id = (int)$_POST['client'];
        $ville = htmlspecialchars($_POST['ville']);
        $prix = (float)$_POST['prix'];
        
        // التحقق من صحة البيانات
        if ($client_id <= 0 || empty($ville) || $prix <= 0) {
            throw new PDOException("Données invalides");
        }
        
        $update_stmt = $pdo->prepare("UPDATE client_prices SET client_id = ?, ville = ?, prix = ? WHERE id = ?");
        $result = $update_stmt->execute([$client_id, $ville, $prix, $price_id]);
        
        if (!$result) {
            throw new PDOException("Erreur lors de la mise à jour du prix");
        }
        
        $_SESSION['success_message'] = "Prix client mis à jour avec succès";
        header('Location: remises.php');
        exit();
    }

} catch (PDOException $e) {
    $_SESSION['error_message'] = "Erreur: " . $e->getMessage();
    header('Location: remises.php');
    exit();
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Modifier Prix Client - My Delivery</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #4361ee;
            --secondary-color: #3a0ca3;
            --accent-color: #4cc9f0;
            --light-color: #f8f9fa;
            --dark-color: #212529;
            --success-color: #4bb543;
            --error-color: #ff3333;
            --warning-color: #f8d210;
            --admin-color: #6a1b9a;
            --border-radius: 12px;
            --box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: #f5f7fa;
            color: var(--dark-color);
            padding: 30px;
        }
        
        .edit-container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--box-shadow);
            padding: 30px;
        }
        
        .page-title {
            font-size: 1.8rem;
            font-weight: 600;
            margin-bottom: 30px;
            color: var(--admin-color);
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: #555;
        }
        
        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid #ddd;
            border-radius: var(--border-radius);
            font-family: 'Poppins', sans-serif;
            font-size: 1rem;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: var(--admin-color);
            outline: none;
            box-shadow: 0 0 0 3px rgba(106, 27, 154, 0.1);
        }
        
        .btn {
            display: inline-block;
            padding: 12px 25px;
            background: var(--admin-color);
            color: white;
            border: none;
            border-radius: var(--border-radius);
            font-family: 'Poppins', sans-serif;
            font-size: 1rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 10px;
        }
        
        .btn:hover {
            background: #5a189a;
            transform: translateY(-2px);
        }
        
        .btn-back {
            background: #6c757d;
            margin-right: 10px;
        }
        
        .btn-back:hover {
            background: #5a6268;
        }
        
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background-color: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            border: 1px solid #f5c6cb;
        }
        
        @media (max-width: 768px) {
            body {
                padding: 15px;
            }
            
            .edit-container {
                padding: 20px;
            }
        }
    </style>
</head>
<body>
    <div class="edit-container">
        <h1 class="page-title">
            <i class="fas fa-edit"></i> Modifier Prix Client
        </h1>
        
        <?php if (isset($_SESSION['error_message'])): ?>
            <div class="alert-error">
                <i class="fas fa-exclamation-circle"></i> <?= $_SESSION['error_message'] ?>
            </div>
            <?php unset($_SESSION['error_message']); ?>
        <?php endif; ?>
        
        <form method="POST" action="">
            <div class="form-group">
                <label for="client">Client</label>
                <select class="form-control" id="client" name="client" required>
                    <option value="">Sélectionner un client</option>
                    <?php foreach ($clients as $client): ?>
                        <option value="<?= htmlspecialchars($client['id']) ?>" <?= $client['id'] == $price['client_id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($client['nom']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-group">
                <label for="ville">Ville</label>
                <select class="form-control" id="ville" name="ville" required>
                    <option value="">Sélectionner une ville</option>
                    <?php foreach ($villes as $ville): ?>
                        <option value="<?= htmlspecialchars($ville['nom_ville']) ?>" <?= $ville['nom_ville'] == $price['ville'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($ville['nom_ville']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-group">
                <label for="prix">Prix (DH)</label>
                <input type="number" class="form-control" id="prix" name="prix" 
                       step="0.01" min="0" required 
                       value="<?= htmlspecialchars($price['prix']) ?>"
                       placeholder="Prix spécial pour le client">
            </div>
            
            <div class="form-group">
                <a href="remises.php" class="btn btn-back">
                    <i class="fas fa-arrow-left"></i> Retour
                </a>
                <button type="submit" name="update_price" class="btn">
                    <i class="fas fa-save"></i> Enregistrer
                </button>
            </div>
        </form>
    </div>
</body>
</html>