<?php
// Activation des erreurs pour le développement
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// Vérification de la connexion et des permissions
if (!isset($_SESSION['user']['id'])) {
    $_SESSION['login_error'] = "Vous devez vous connecter d'abord";
    header('Location: login.php');
    exit();
}

if ($_SESSION['user']['role'] !== 'admin') {
    die("Accès refusé : Vous n'avez pas les permissions nécessaires.");
}

require 'db.php';

// Initialisation des messages
$message = '';
$error = '';

try {
    // Ajout d'une nouvelle ville
    if (isset($_POST['add_ville'])) {
        $nom_ville = trim($_POST['nom_ville']);
        $frais_livraison = floatval($_POST['frais_livraison']);
        
        if (empty($nom_ville) || $frais_livraison <= 0) {
            $error = "Veuillez remplir tous les champs correctement";
        } else {
            $stmt = $pdo->prepare("INSERT INTO villes (nom_ville, frais_livraison) VALUES (?, ?)");
            $stmt->execute([$nom_ville, $frais_livraison]);
            $message = "Ville ajoutée avec succès";
        }
    }
    
    // Modification d'une ville
    if (isset($_POST['update_ville'])) {
        $id = intval($_POST['id']);
        $nom_ville = trim($_POST['nom_ville']);
        $frais_livraison = floatval($_POST['frais_livraison']);
        
        if (empty($nom_ville) || $frais_livraison <= 0) {
            $error = "Veuillez remplir tous les champs correctement";
        } else {
            $stmt = $pdo->prepare("UPDATE villes SET nom_ville = ?, frais_livraison = ? WHERE id = ?");
            $stmt->execute([$nom_ville, $frais_livraison, $id]);
            $message = "Ville mise à jour avec succès";
        }
    }
    
    // Suppression d'une ville
    if (isset($_GET['delete'])) {
        $id = intval($_GET['delete']);
        
        // Vérification des colis associés
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM colis WHERE ville_id = ?");
        $stmt->execute([$id]);
        $count = $stmt->fetchColumn();
        
        if ($count > 0) {
            $error = "Impossible de supprimer cette ville car elle est associée à des colis";
        } else {
            $stmt = $pdo->prepare("DELETE FROM villes WHERE id = ?");
            $stmt->execute([$id]);
            $message = "Ville supprimée avec succès";
        }
    }
    
    // Ajout des villes marocaines principales
    if (isset($_POST['add_villes_maroc'])) {
        $villes_maroc = [
            ['nom' => 'Casablanca', 'prix' => 30],
            ['nom' => 'Rabat', 'prix' => 25],
            ['nom' => 'Fès', 'prix' => 40],
            ['nom' => 'Tanger', 'prix' => 50],
            ['nom' => 'Marrakech', 'prix' => 45],
            ['nom' => 'Meknès', 'prix' => 35],
            ['nom' => 'Agadir', 'prix' => 60],
            ['nom' => 'Oujda', 'prix' => 70],
            ['nom' => 'Kénitra', 'prix' => 30],
            ['nom' => 'Tétouan', 'prix' => 55],
            ['nom' => 'Safi', 'prix' => 50],
            ['nom' => 'Mohammedia', 'prix' => 25],
            ['nom' => 'Khouribga', 'prix' => 40],
            ['nom' => 'El Jadida', 'prix' => 40],
            ['nom' => 'Nador', 'prix' => 65],
            ['nom' => 'Taza', 'prix' => 50],
            ['nom' => 'Settat', 'prix' => 35],
            ['nom' => 'Berrechid', 'prix' => 30],
            ['nom' => 'Khemisset', 'prix' => 35],
            ['nom' => 'Inezgane', 'prix' => 60]
        ];

        $ajoutees = 0;
        $existantes = 0;
        
        foreach ($villes_maroc as $ville) {
            // Vérifier si la ville existe déjà
            $stmt = $pdo->prepare("SELECT id FROM villes WHERE nom_ville = ?");
            $stmt->execute([$ville['nom']]);
            
            if ($stmt->rowCount() == 0) {
                $insert = $pdo->prepare("INSERT INTO villes (nom_ville, frais_livraison) VALUES (?, ?)");
                $insert->execute([$ville['nom'], $ville['prix']]);
                $ajoutees++;
            } else {
                $existantes++;
            }
        }
        
        $message = "Opération terminée : $ajoutees villes ajoutées, $existantes villes existaient déjà";
    }

    // Récupération de toutes les villes
    $villes = $pdo->query("SELECT * FROM villes ORDER BY nom_ville")->fetchAll();
    
} catch (PDOException $e) {
    $error = "Erreur de base de données : " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion Villes & Tarifs - My Delivery</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #4361ee;
            --secondary-color: #3a0ca3;
            --accent-color: #4cc9f0;
            --light-color: #f8f9fa;
            --dark-color: #212529;
            --success-color: #4bb543;
            --error-color: #ff3333;
            --warning-color: #f8d210;
            --admin-color: #6a1b9a;
            --border-radius: 12px;
            --box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: #f5f7fa;
            color: var(--dark-color);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        .admin-sidebar {
            width: 280px;
            background: var(--admin-color);
            color: white;
            padding: 20px 0;
        }
        
        .admin-sidebar-header {
            padding: 0 20px 20px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            margin-bottom: 20px;
        }
        
        .admin-logo {
            font-size: 1.5rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .admin-logo i {
            font-size: 2rem;
        }
        
        .admin-menu {
            list-style: none;
        }
        
        .admin-menu li a {
            display: flex;
            align-items: center;
            padding: 15px 25px;
            color: rgba(255,255,255,0.8);
            text-decoration: none;
            transition: all 0.3s ease;
            gap: 12px;
            font-size: 0.95rem;
        }
        
        .admin-menu li a:hover, 
        .admin-menu li a.active {
            background: rgba(255,255,255,0.1);
            color: white;
            border-left: 3px solid white;
        }
        
        .admin-menu li a i {
            width: 20px;
            text-align: center;
            font-size: 1.1rem;
        }
        
        .admin-main {
            flex: 1;
            padding: 30px;
            background: #f5f7fa;
        }
        
        .admin-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid #eee;
        }
        
        .admin-title {
            font-size: 1.8rem;
            font-weight: 600;
            color: var(--dark-color);
        }
        
        .admin-user {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .admin-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: var(--primary-color);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
        }
        
        .admin-section {
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--box-shadow);
            padding: 25px;
            margin-bottom: 30px;
        }
        
        .section-title {
            font-size: 1.2rem;
            font-weight: 600;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            color: var(--admin-color);
        }
        
        .section-title i {
            font-size: 1.4rem;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }
        
        .data-table th {
            text-align: left;
            padding: 12px 15px;
            background: #f8f9fa;
            font-weight: 500;
            color: #555;
        }
        
        .data-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #eee;
        }
        
        .data-table tr:last-child td {
            border-bottom: none;
        }
        
        .action-btn {
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 0.8rem;
            text-decoration: none;
            margin-right: 5px;
        }
        
        .edit-btn {
            background: rgba(248, 210, 16, 0.2);
            color: #b38f00;
        }
        
        .edit-btn:hover {
            background: rgba(248, 210, 16, 0.3);
        }
        
        .delete-btn {
            background: rgba(255, 51, 51, 0.2);
            color: var(--error-color);
        }
        
        .delete-btn:hover {
            background: rgba(255, 51, 51, 0.3);
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        .form-control {
            width: 100%;
            padding: 10px 15px;
            border: 1px solid #ddd;
            border-radius: var(--border-radius);
            font-family: 'Poppins', sans-serif;
        }
        
        .btn {
            padding: 10px 20px;
            border-radius: var(--border-radius);
            border: none;
            cursor: pointer;
            font-family: 'Poppins', sans-serif;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        
        .btn-primary {
            background: var(--admin-color);
            color: white;
        }
        
        .btn-primary:hover {
            background: #5a148a;
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #5a6268;
        }
        
        .btn-success {
            background: var(--success-color);
            color: white;
        }
        
        .btn-success:hover {
            background: #3a9a32;
        }
        
        .alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 20px;
        }
        
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-danger {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        
        .modal-content {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            width: 100%;
            max-width: 500px;
            box-shadow: var(--box-shadow);
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .modal-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: var(--admin-color);
        }
        
        .close-btn {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #777;
        }
        
        .text-right {
            text-align: right;
        }
        
        @media (max-width: 992px) {
            .admin-container {
                flex-direction: column;
            }
            
            .admin-sidebar {
                width: 100%;
                height: auto;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Barre latérale -->
        <div class="admin-sidebar">
            <div class="admin-sidebar-header">
                <div class="admin-logo">
                    <i class="fas fa-user-shield"></i>
                    <span>Admin Panel</span>
                </div>
            </div>
            <ul class="admin-menu">
                <li><a href="index.php"><i class="fas fa-tachometer-alt"></i> Tableau de Bord</a></li>
                <li><a href="admin_users.php"><i class="fas fa-users"></i> Gestion Utilisateurs</a></li>
                <li><a href="admin_colis.php"><i class="fas fa-boxes"></i> Gestion Colis</a></li>
                <li><a href="admin_villes.php" class="active"><i class="fas fa-city"></i> Gestion Villes & Tarifs</a></li>
                <li><a href="admin_commissions.php"><i class="fas fa-money-bill-wave"></i> Commissions Livreurs</a></li>
                <li><a href="remises.php"><i class="fas fa-tag"></i> Remise Clients</a></li>
               
                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a></li>
            </ul>
        </div>
        
        <!-- Contenu principal -->
        <div class="admin-main">
            <div class="admin-header">
                <h1 class="admin-title"><i class="fas fa-city"></i> Gestion Villes & Tarifs de Livraison</h1>
                <div class="admin-user">
                    <div class="admin-avatar">
                        <?= substr($_SESSION['user']['nom'] ?? 'A', 0, 1) ?>
                    </div>
                    <span><?= $_SESSION['user']['nom'] ?? 'Admin' ?></span>
                </div>
            </div>
            
            <!-- Messages d'alerte -->
            <?php if ($message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?= $message ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle"></i> <?= $error ?>
                </div>
            <?php endif; ?>
            
            <!-- Section d'ajout rapide -->
            <div class="admin-section">
                <h3 class="section-title">
                    <i class="fas fa-bolt"></i> Ajout Rapide
                </h3>
                
                <form method="POST" action="" onsubmit="return confirm('Voulez-vous vraiment ajouter les principales villes marocaines ?')">
                    <p>Ajouter automatiquement les 20 principales villes marocaines avec des tarifs de livraison par défaut :</p>
                    <button type="submit" name="add_villes_maroc" class="btn btn-success">
                        <i class="fas fa-flag"></i> Ajouter Villes Marocaines
                    </button>
                    <small class="text-muted">Les villes existantes ne seront pas dupliquées</small>
                </form>
            </div>
            
            <!-- Formulaire d'ajout de ville -->
            <div class="admin-section">
                <h3 class="section-title">
                    <i class="fas fa-plus-circle"></i> Ajouter une nouvelle ville
                </h3>
                
                <form method="POST" action="">
                    <div class="form-group">
                        <label for="nom_ville">Nom de la ville</label>
                        <input type="text" id="nom_ville" name="nom_ville" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="frais_livraison">Frais de livraison (DH)</label>
                        <input type="number" id="frais_livraison" name="frais_livraison" class="form-control" min="0" step="0.01" required>
                    </div>
                    
                    <div class="text-right">
                        <button type="submit" name="add_ville" class="btn btn-primary">
                            <i class="fas fa-save"></i> Enregistrer
                        </button>
                    </div>
                </form>
            </div>
            
            <!-- Liste des villes existantes -->
            <div class="admin-section">
                <h3 class="section-title">
                    <i class="fas fa-list"></i> Liste des villes
                </h3>
                
                <?php if (empty($villes)): ?>
                    <p>Aucune ville enregistrée pour le moment.</p>
                <?php else: ?>
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Ville</th>
                                <th>Frais (DH)</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($villes as $ville): ?>
                                <tr>
                                    <td><?= $ville['id'] ?></td>
                                    <td><?= htmlspecialchars($ville['nom_ville']) ?></td>
                                    <td><?= number_format($ville['frais_livraison'], 2) ?></td>
                                    <td>
                                        <button onclick="openEditModal(<?= $ville['id'] ?>, '<?= htmlspecialchars($ville['nom_ville'], ENT_QUOTES) ?>', <?= $ville['frais_livraison'] ?>)" 
                                                class="action-btn edit-btn">
                                            <i class="fas fa-edit"></i> Modifier
                                        </button>
                                        <a href="admin_villes.php?delete=<?= $ville['id'] ?>" 
                                           class="action-btn delete-btn"
                                           onclick="return confirm('Êtes-vous sûr de vouloir supprimer cette ville ?')">
                                            <i class="fas fa-trash"></i> Supprimer
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Modal de modification -->
    <div id="editModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title"><i class="fas fa-edit"></i> Modifier la ville</h3>
                <button class="close-btn" onclick="closeEditModal()">&times;</button>
            </div>
            
            <form method="POST" action="">
                <input type="hidden" id="edit_id" name="id">
                
                <div class="form-group">
                    <label for="edit_nom_ville">Nom de la ville</label>
                    <input type="text" id="edit_nom_ville" name="nom_ville" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label for="edit_frais_livraison">Frais de livraison (DH)</label>
                    <input type="number" id="edit_frais_livraison" name="frais_livraison" class="form-control" min="0" step="0.01" required>
                </div>
                
                <div class="text-right">
                    <button type="button" class="btn btn-secondary" onclick="closeEditModal()">
                        <i class="fas fa-times"></i> Annuler
                    </button>
                    <button type="submit" name="update_ville" class="btn btn-primary">
                        <i class="fas fa-save"></i> Enregistrer
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
        // Ouvrir le modal de modification
        function openEditModal(id, nom_ville, frais_livraison) {
            document.getElementById('edit_id').value = id;
            document.getElementById('edit_nom_ville').value = nom_ville;
            document.getElementById('edit_frais_livraison').value = frais_livraison;
            document.getElementById('editModal').style.display = 'flex';
        }
        
        // Fermer le modal de modification
        function closeEditModal() {
            document.getElementById('editModal').style.display = 'none';
        }
        
        // Fermer le modal en cliquant à l'extérieur
        window.onclick = function(event) {
            const modal = document.getElementById('editModal');
            if (event.target === modal) {
                closeEditModal();
            }
        }
    </script>
</body>
</html>