<?php
session_start();
require 'db.php';

// Vérification des privilèges admin
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header('Location: login.php');
    exit();
}

// Récupération des utilisateurs avec filtre optionnel
$role_filter = isset($_GET['role_filter']) && in_array($_GET['role_filter'], ['client', 'livreur', 'admin']) ? $_GET['role_filter'] : null;

if ($role_filter) {
    $stmt = $pdo->prepare("SELECT * FROM users WHERE role = ? ORDER BY id DESC");
    $stmt->execute([$role_filter]);
} else {
    $stmt = $pdo->prepare("SELECT * FROM users ORDER BY id DESC");
    $stmt->execute();
}
$users = $stmt->fetchAll();

// Récupération des villes depuis la base de données
$stmt_villes = $pdo->query("SELECT nom_ville FROM villes ORDER BY nom_ville");
$villes = $stmt_villes->fetchAll(PDO::FETCH_COLUMN, 0);

// Traitement des mises à jour
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $user_id = $_POST['user_id'];
        $action = $_POST['action'];
        
        if ($action === 'update_role_status') {
            // Mise à jour rôle et statut
            if (isset($_POST['statut']) && isset($_POST['role'])) {
                $statut = $_POST['statut'];
                $role = $_POST['role'];

                $allowed_status = ['en_attente', 'actif', 'annulé'];
                $allowed_roles = ['client', 'livreur', 'admin'];
                
                if (in_array($statut, $allowed_status) && in_array($role, $allowed_roles)) {
                    $stmt = $pdo->prepare("UPDATE users SET statut = ?, role = ? WHERE id = ?");
                    $stmt->execute([$statut, $role, $user_id]);
                    
                    $_SESSION['admin_message'] = "Statut et rôle mis à jour avec succès!";
                }
            }
        } elseif ($action === 'update_profile') {
            // Mise à jour profil
            $email = $_POST['email'] ?? '';
            $telephone = $_POST['telephone'] ?? '';
            $bank_account = $_POST['bank_account'] ?? '';
            
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $_SESSION['admin_message'] = "Email invalide";
            } else {
                $stmt = $pdo->prepare("UPDATE users SET email = ?, telephone = ?, bank_account = ? WHERE id = ?");
                $stmt->execute([$email, $telephone, $bank_account, $user_id]);
                
                // Gestion des villes sélectionnées
                // Supprimer les anciennes villes sélectionnées
                $pdo->prepare("DELETE FROM user_villes WHERE user_id = ?")->execute([$user_id]);
                
                // Ajouter les nouvelles villes sélectionnées
                if (!empty($_POST['villes'])) {
                    $stmt_insert = $pdo->prepare("INSERT INTO user_villes (user_id, ville) VALUES (?, ?)");
                    foreach ($_POST['villes'] as $ville) {
                        $stmt_insert->execute([$user_id, $ville]);
                    }
                }
                
                $_SESSION['admin_message'] = "Profil utilisateur mis à jour!";
            }
        } elseif ($action === 'delete_user') {
            // Suppression utilisateur
            if ($_SESSION['user']['id'] != $user_id) {
                // Supprimer d'abord les villes associées
                $pdo->prepare("DELETE FROM user_villes WHERE user_id = ?")->execute([$user_id]);
                
                // Puis supprimer l'utilisateur
                $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $_SESSION['admin_message'] = "Utilisateur supprimé avec succès!";
            } else {
                $_SESSION['admin_message'] = "Vous ne pouvez pas supprimer votre propre compte!";
            }
        }
    }
    header("Location: admin_users.php".($role_filter ? "?role_filter=".$role_filter : ""));
    exit();
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion Utilisateurs - Admin</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #4361ee;
            --secondary-color: #3a0ca3;
            --accent-color: #4cc9f0;
            --light-color: #f8f9fa;
            --dark-color: #212529;
            --success-color: #4bb543;
            --error-color: #ff3333;
            --warning-color: #f8d210;
            --admin-color: #6a1b9a;
            --border-radius: 12px;
            --box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: #f5f7fa;
            color: var(--dark-color);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        .admin-sidebar {
            width: 280px;
            background: var(--admin-color);
            color: white;
            padding: 20px 0;
        }
        
        .admin-sidebar-header {
            padding: 0 20px 20px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            margin-bottom: 20px;
        }
        
        .admin-logo {
            font-size: 1.5rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .admin-menu {
            list-style: none;
            padding: 0;
        }
        
        .admin-menu li a {
            display: flex;
            align-items: center;
            padding: 15px 25px;
            color: rgba(255,255,255,0.8);
            text-decoration: none;
            transition: all 0.3s ease;
            gap: 12px;
            font-size: 0.95rem;
        }
        
        .admin-menu li a:hover, 
        .admin-menu li a.active {
            background: rgba(255,255,255,0.1);
            color: white;
            border-left: 3px solid white;
        }
        
        .admin-main {
            flex: 1;
            padding: 30px;
            background: #f5f7fa;
        }
        
        .admin-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid #eee;
        }
        
        .admin-title {
            font-size: 1.8rem;
            font-weight: 600;
            color: var(--dark-color);
        }
        
        .admin-section {
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--box-shadow);
            padding: 25px;
            margin-bottom: 30px;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .data-table th {
            text-align: left;
            padding: 12px 15px;
            background: #f8f9fa;
            font-weight: 500;
            color: #555;
        }
        
        .data-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #eee;
            vertical-align: middle;
        }
        
        .status-badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 5px;
        }
        
        .status-pending {
            background: rgba(248, 210, 16, 0.2);
            color: #b38f00;
        }
        
        .status-active {
            background: rgba(75, 181, 67, 0.2);
            color: var(--success-color);
        }
        
        .status-cancelled {
            background: rgba(255, 51, 51, 0.2);
            color: var(--error-color);
        }
        
        .role-badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 5px;
        }
        
        .role-client {
            background: rgba(67, 97, 238, 0.2);
            color: var(--primary-color);
        }
        
        .role-livreur {
            background: rgba(76, 201, 240, 0.2);
            color: var(--accent-color);
        }
        
        .role-admin {
            background: rgba(106, 27, 154, 0.2);
            color: var(--admin-color);
        }
        
        .action-form {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        
        .action-dropdown {
            position: relative;
        }
        
        .action-btn {
            padding: 6px 12px;
            border-radius: var(--border-radius);
            font-size: 0.85rem;
            display: flex;
            align-items: center;
            gap: 5px;
            transition: all 0.3s ease;
        }
        
        .role-btn {
            background-color: rgba(67, 97, 238, 0.1);
            border: 1px solid rgba(67, 97, 238, 0.3);
            color: var(--primary-color);
        }
        
        .status-btn {
            background-color: rgba(75, 181, 67, 0.1);
            border: 1px solid rgba(75, 181, 67, 0.3);
            color: var(--success-color);
        }
        
        .save-btn {
            background-color: var(--success-color);
            color: white;
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            border: none;
        }
        
        .delete-btn {
            background-color: var(--error-color);
            color: white;
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            border: none;
        }
        
        .dropdown-menu {
            border: none;
            box-shadow: var(--box-shadow);
            border-radius: var(--border-radius);
            padding: 5px;
        }
        
        .dropdown-item {
            padding: 8px 12px;
            border-radius: 4px;
            font-size: 0.85rem;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .dropdown-item:hover {
            background-color: rgba(106, 27, 154, 0.1);
        }
        
        .message-alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .message-success {
            background: rgba(75, 181, 67, 0.1);
            color: var(--success-color);
            border-left: 4px solid var(--success-color);
        }
        
        .message-error {
            background: rgba(255, 51, 51, 0.1);
            color: var(--error-color);
            border-left: 4px solid var(--error-color);
        }
        
        /* Styles pour le modal */
        .modal-content {
            border-radius: var(--border-radius);
            border: none;
            box-shadow: var(--box-shadow);
        }
        
        .modal-header {
            background: var(--admin-color);
            color: white;
            border-radius: var(--border-radius) var(--border-radius) 0 0;
        }
        
        .profile-form .form-group {
            margin-bottom: 1.5rem;
        }
        
        .profile-form label {
            font-weight: 500;
            margin-bottom: 0.5rem;
        }
        
        .view-profile-btn {
            background: rgba(67, 97, 238, 0.1);
            color: var(--primary-color);
            border: none;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 0.85rem;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .view-profile-btn:hover {
            background: rgba(67, 97, 238, 0.2);
        }
        
        .action-buttons {
            display: flex;
            gap: 8px;
        }
        
        /* أنماط إضافية لتصفية الأدوار */
        .role-filter {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 20px;
        }
        
        .role-filter select {
            flex: 1;
            max-width: 250px;
        }
        
        .filter-badge {
            display: inline-flex;
            align-items: center;
            padding: 5px 10px;
            background-color: rgba(106, 27, 154, 0.1);
            color: var(--admin-color);
            border-radius: 20px;
            font-size: 0.9rem;
            margin-right: 10px;
        }
        
        .filter-badge i {
            margin-right: 5px;
        }
        
        /* Styles pour le select multiple */
        select[multiple] {
            min-height: 120px;
        }
        
        select[multiple] option {
            padding: 8px 12px;
        }
        
        .selected-villes {
            margin-top: 10px;
        }
        
        .selected-villes .badge {
            margin-right: 5px;
            margin-bottom: 5px;
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Menu latéral -->
        <div class="admin-sidebar">
            <div class="admin-sidebar-header">
                <div class="admin-logo">
                    <i class="fas fa-user-shield"></i>
                    <span>Admin Panel</span>
                </div>
            </div>
            <ul class="admin-menu">
               <li><a href="index.php"><i class="fas fa-tachometer-alt"></i> Tableau de Bord</a></li>
                <li><a href="admin_users.php" class="active"><i class="fas fa-users"></i> Gestion Utilisateurs</a></li>
                <li><a href="admin_colis.php"><i class="fas fa-boxes"></i> Gestion Colis</a></li>
                <li><a href="admin_villes.php"><i class="fas fa-city"></i> Gestion Villes & Tarifs</a></li>
                <li><a href="admin_commissions.php"><i class="fas fa-money-bill-wave"></i> Commissions Livreurs</a></li>
                <li><a href="remises.php"><i class="fas fa-tag"></i> Remise Clients</a></li>
               
                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a></li>
            </ul>
        </div>
        
        <!-- Contenu principal -->
        <div class="admin-main">
            <div class="admin-header">
                <h1 class="admin-title"><i class="fas fa-users-cog"></i> Gestion des Utilisateurs</h1>
            </div>
            
            <?php if (isset($_SESSION['admin_message'])): ?>
                <div class="message-alert <?= strpos($_SESSION['admin_message'], 'succès') !== false ? 'message-success' : 'message-error' ?>">
                    <i class="fas <?= strpos($_SESSION['admin_message'], 'succès') !== false ? 'fa-check-circle' : 'fa-exclamation-circle' ?>"></i>
                    <?= $_SESSION['admin_message'] ?>
                </div>
                <?php unset($_SESSION['admin_message']); ?>
            <?php endif; ?>
            
            <div class="admin-section">
                <!-- إضافة حقل البحث حسب الأدوار -->
                <div class="row mb-4">
                    <div class="col-md-4">
                        <form method="GET" class="d-flex">
                            <select name="role_filter" class="form-select me-2" onchange="this.form.submit()">
                                <option value="">Tous les rôles</option>
                                <option value="client" <?= isset($_GET['role_filter']) && $_GET['role_filter'] == 'client' ? 'selected' : '' ?>>Clients</option>
                                <option value="livreur" <?= isset($_GET['role_filter']) && $_GET['role_filter'] == 'livreur' ? 'selected' : '' ?>>Livreurs</option>
                                <option value="admin" <?= isset($_GET['role_filter']) && $_GET['role_filter'] == 'admin' ? 'selected' : '' ?>>Administrateurs</option>
                            </select>
                            <button type="button" class="btn btn-outline-secondary" onclick="window.location.href='admin_users.php'">
                                <i class="fas fa-times"></i>
                            </button>
                        </form>
                    </div>
                </div>
                
                <?php if ($role_filter): ?>
                    <div class="mb-3">
                        <span class="filter-badge">
                            <i class="fas fa-filter"></i>
                            Filtre actif: <?= ucfirst($role_filter) ?>
                            <a href="admin_users.php" class="ms-2 text-danger">
                                <i class="fas fa-times"></i>
                            </a>
                        </span>
                    </div>
                <?php endif; ?>
                
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Nom</th>
                            <th>Email</th>
                            <th>Rôle</th>
                            <th>Statut</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($users as $user): 
                        // Récupérer les villes de l'utilisateur
                        $stmt_user_villes = $pdo->prepare("SELECT ville FROM user_villes WHERE user_id = ?");
                        $stmt_user_villes->execute([$user['id']]);
                        $user_villes = $stmt_user_villes->fetchAll(PDO::FETCH_COLUMN, 0);
                    ?>
                        <tr>
                            <td><?= $user['id'] ?></td>
                            <td><?= htmlspecialchars($user['nom']) ?></td>
                            <td><?= htmlspecialchars($user['email']) ?></td>
                            <td>
                                <span class="role-badge role-<?= $user['role'] ?>">
                                    <i class="fas <?= 
                                        $user['role'] == 'client' ? 'fa-user' : 
                                        ($user['role'] == 'livreur' ? 'fa-truck' : 'fa-user-shield') 
                                    ?>"></i>
                                    <?= ucfirst($user['role']) ?>
                                </span>
                            </td>
                            <td>
                                <span class="status-badge status-<?= str_replace(' ', '-', strtolower($user['statut'])) ?>">
                                    <i class="fas <?= 
                                        $user['statut'] == 'actif' ? 'fa-check-circle' : 
                                        ($user['statut'] == 'annulé' ? 'fa-times-circle' : 'fa-clock') 
                                    ?>"></i>
                                    <?= ucfirst(str_replace('_', ' ', $user['statut'])) ?>
                                </span>
                            </td>
                            <td>
                                <div class="d-flex gap-2">
                                    <!-- Bouton pour voir/modifier le profil -->
                                    <button class="view-profile-btn" data-bs-toggle="modal" data-bs-target="#profileModal" 
                                        data-user-id="<?= $user['id'] ?>"
                                        data-user-nom="<?= htmlspecialchars($user['nom']) ?>"
                                        data-user-prenom="<?= htmlspecialchars($user['prenom']) ?>"
                                        data-user-email="<?= htmlspecialchars($user['email']) ?>"
                                        data-user-telephone="<?= htmlspecialchars($user['telephone'] ?? '') ?>"
                                        data-user-bank="<?= htmlspecialchars($user['bank_account'] ?? '') ?>"
                                        data-user-villes="<?= htmlspecialchars(implode(',', $user_villes)) ?>">
                                        <i class="fas fa-user-edit"></i> Profil
                                    </button>
                                    
                                    <!-- Formulaire de modification rôle/statut -->
                                    <form method="POST" class="action-form">
                                        <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                        <input type="hidden" name="action" value="update_role_status">
                                        
                                        <!-- Menu déroulant Rôle -->
                                        <div class="action-dropdown">
                                            <button class="action-btn role-btn dropdown-toggle" type="button" 
                                                data-bs-toggle="dropdown" aria-expanded="false">
                                                <i class="fas <?= 
                                                    $user['role'] == 'client' ? 'fa-user' : 
                                                    ($user['role'] == 'livreur' ? 'fa-truck' : 'fa-user-shield') 
                                                ?>"></i>
                                                <span><?= ucfirst($user['role']) ?></span>
                                            </button>
                                            <ul class="dropdown-menu">
                                                <li>
                                                    <button class="dropdown-item" type="button" onclick="setRole(this, 'client')">
                                                        <i class="fas fa-user text-primary"></i> Client
                                                    </button>
                                                </li>
                                                <li>
                                                    <button class="dropdown-item" type="button" onclick="setRole(this, 'livreur')">
                                                        <i class="fas fa-truck text-info"></i> Livreur
                                                    </button>
                                                </li>
                                                <?php if ($_SESSION['user']['id'] != $user['id']): ?>
                                                <li>
                                                    <button class="dropdown-item" type="button" onclick="setRole(this, 'admin')">
                                                        <i class="fas fa-user-shield text-purple"></i> Admin
                                                    </button>
                                                </li>
                                                <?php endif; ?>
                                            </ul>
                                            <input type="hidden" name="role" value="<?= $user['role'] ?>">
                                        </div>
                                        
                                        <!-- Menu déroulant Statut -->
                                        <div class="action-dropdown">
                                            <button class="action-btn status-btn dropdown-toggle" type="button" 
                                                data-bs-toggle="dropdown" aria-expanded="false">
                                                <i class="fas <?= 
                                                    $user['statut'] == 'actif' ? 'fa-check-circle' : 
                                                    ($user['statut'] == 'annulé' ? 'fa-times-circle' : 'fa-clock') 
                                                ?>"></i>
                                                <span><?= ucfirst(str_replace('_', ' ', $user['statut'])) ?></span>
                                            </button>
                                            <ul class="dropdown-menu">
                                                <li>
                                                    <button class="dropdown-item" type="button" onclick="setStatus(this, 'en_attente')">
                                                        <i class="fas fa-clock text-warning"></i> En attente
                                                    </button>
                                                </li>
                                                <li>
                                                    <button class="dropdown-item" type="button" onclick="setStatus(this, 'actif')">
                                                        <i class="fas fa-check-circle text-success"></i> Activer
                                                    </button>
                                                </li>
                                                <li>
                                                    <button class="dropdown-item" type="button" onclick="setStatus(this, 'annulé')">
                                                        <i class="fas fa-times-circle text-danger"></i> Désactiver
                                                    </button>
                                                </li>
                                            </ul>
                                            <input type="hidden" name="statut" value="<?= $user['statut'] ?>">
                                        </div>
                                        
                                        <!-- Boutons Enregistrer et Supprimer -->
                                        <div class="action-buttons">
                                            <button type="submit" class="save-btn" title="Enregistrer">
                                                <i class="fas fa-save"></i>
                                            </button>
                                            
                                            <?php if ($_SESSION['user']['id'] != $user['id']): ?>
                                            <button type="button" class="delete-btn" title="Supprimer" 
                                                onclick="confirmDelete(<?= $user['id'] ?>, '<?= htmlspecialchars($user['nom']) ?>')">
                                                <i class="fas fa-trash-alt"></i>
                                            </button>
                                            <?php endif; ?>
                                        </div>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Modal pour voir/modifier le profil -->
    <div class="modal fade" id="profileModal" tabindex="-1" aria-labelledby="profileModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="profileModalLabel">Profil Utilisateur</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form id="profileForm" method="POST" class="profile-form">
                        <input type="hidden" name="user_id" id="modalUserId">
                        <input type="hidden" name="action" value="update_profile">
                        
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="modalNom">Nom</label>
                                    <input type="text" id="modalNom" class="form-control" disabled>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="modalPrenom">Prénom</label>
                                    <input type="text" id="modalPrenom" class="form-control" disabled>
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-group mb-3">
                            <label for="modalEmail">Email *</label>
                            <input type="email" id="modalEmail" name="email" class="form-control" required>
                        </div>
                        
                        <div class="form-group mb-3">
                            <label for="modalTelephone">Téléphone *</label>
                            <input type="tel" id="modalTelephone" name="telephone" class="form-control" required>
                        </div>
                        
                        <div class="form-group mb-3">
                            <label for="modalBank">Compte Bancaire</label>
                            <input type="text" id="modalBank" name="bank_account" class="form-control">
                        </div>
                        
                        <div class="form-group mb-3">
                            <label for="modalVilles">Villes</label>
                            <select id="modalVilles" name="villes[]" class="form-select" multiple="multiple" style="height: auto;">
                                <?php foreach ($villes as $ville): ?>
                                    <option value="<?= htmlspecialchars($ville) ?>"><?= htmlspecialchars($ville) ?></option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Maintenez Ctrl pour sélectionner plusieurs villes</small>
                            <div id="selectedVilles" class="selected-villes mt-2"></div>
                        </div>
                        
                        <div class="d-flex justify-content-end">
                            <button type="button" class="btn btn-secondary me-2" data-bs-dismiss="modal">Annuler</button>
                            <button type="submit" class="btn btn-primary">Enregistrer</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Formulaire caché pour la suppression -->
    <form id="deleteForm" method="POST" style="display: none;">
        <input type="hidden" name="user_id" id="deleteUserId">
        <input type="hidden" name="action" value="delete_user">
    </form>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Fonction pour mettre à jour le rôle
        function setRole(element, role) {
            const form = element.closest('.action-form');
            const input = form.querySelector('input[name="role"]');
            const button = form.querySelector('.role-btn');
            
            input.value = role;
            
            const iconClass = role === 'client' ? 'fa-user' : 
                            (role === 'livreur' ? 'fa-truck' : 'fa-user-shield');
            const text = role.charAt(0).toUpperCase() + role.slice(1);
            
            button.innerHTML = `<i class="fas ${iconClass}"></i><span>${text}</span>`;
        }
        
        // Fonction pour mettre à jour le statut
        function setStatus(element, status) {
            const form = element.closest('.action-form');
            const input = form.querySelector('input[name="statut"]');
            const button = form.querySelector('.status-btn');
            
            input.value = status;
            
            const iconClass = status === 'actif' ? 'fa-check-circle' : 
                            (status === 'annulé' ? 'fa-times-circle' : 'fa-clock');
            const text = status === 'en_attente' ? 'En attente' : 
                        (status === 'actif' ? 'Activer' : 'Désactiver');
            
            button.innerHTML = `<i class="fas ${iconClass}"></i><span>${text}</span>`;
        }
        
        // Fonction pour confirmer la suppression
        function confirmDelete(userId, userName) {
            if (confirm(`Êtes-vous sûr de vouloir supprimer l'utilisateur "${userName}"? Cette action est irréversible.`)) {
                document.getElementById('deleteUserId').value = userId;
                document.getElementById('deleteForm').submit();
            }
        }
        
        // Initialisation du modal
        document.addEventListener('DOMContentLoaded', function() {
            var profileModal = document.getElementById('profileModal');
            if (profileModal) {
                profileModal.addEventListener('show.bs.modal', function(event) {
                    var button = event.relatedTarget;
                    
                    document.getElementById('modalUserId').value = button.getAttribute('data-user-id');
                    document.getElementById('modalNom').value = button.getAttribute('data-user-nom');
                    document.getElementById('modalPrenom').value = button.getAttribute('data-user-prenom');
                    document.getElementById('modalEmail').value = button.getAttribute('data-user-email');
                    document.getElementById('modalTelephone').value = button.getAttribute('data-user-telephone');
                    document.getElementById('modalBank').value = button.getAttribute('data-user-bank');
                    
                    // Récupérer les villes sélectionnées
                    var villes = button.getAttribute('data-user-villes');
                    if (villes) {
                        var villesArray = villes.split(',');
                        var select = document.getElementById('modalVilles');
                        
                        // Réinitialiser la sélection
                        Array.from(select.options).forEach(option => {
                            option.selected = false;
                        });
                        
                        // Sélectionner les villes de l'utilisateur
                        Array.from(select.options).forEach(option => {
                            if (villesArray.includes(option.value)) {
                                option.selected = true;
                            }
                        });
                        
                        // Afficher les villes sélectionnées
                        updateSelectedVillesDisplay();
                    }
                });
            }
            
            // Mettre à jour l'affichage des villes sélectionnées
            var villesSelect = document.getElementById('modalVilles');
            if (villesSelect) {
                villesSelect.addEventListener('change', updateSelectedVillesDisplay);
            }
        });
        
        // Fonction pour afficher les villes sélectionnées
        function updateSelectedVillesDisplay() {
            var select = document.getElementById('modalVilles');
            var selectedContainer = document.getElementById('selectedVilles');
            
            if (!select || !selectedContainer) return;
            
            var selectedOptions = Array.from(select.selectedOptions);
            selectedContainer.innerHTML = '';
            
            if (selectedOptions.length > 0) {
                var title = document.createElement('p');
                title.className = 'mb-2 fw-bold';
                title.textContent = 'Villes sélectionnées:';
                selectedContainer.appendChild(title);
                
                var badgesContainer = document.createElement('div');
                selectedOptions.forEach(option => {
                    var badge = document.createElement('span');
                    badge.className = 'badge bg-primary me-2 mb-2';
                    badge.textContent = option.text;
                    badgesContainer.appendChild(badge);
                });
                selectedContainer.appendChild(badgesContainer);
            }
        }
    </script>
</body>
</html>